/*
 * \file memoryusageobserver.cpp
 *
 * This class represents a Singleton for observing memory usage.
 */
#include "memoryusageobserver.hpp"

MemoryUsageObserver* MemoryUsageObserver::instance = NULL;

/**
 * Constructor. Do not use this function. Use getInstance() instead.
 *
 * \return memory usage observer instance
 */
MemoryUsageObserver::MemoryUsageObserver() {
  instance = NULL;
  maximumSize = 0;
  totalSize = 0;
}

/**
 * Destructor. Better use purgeInstance().
 */
MemoryUsageObserver::~MemoryUsageObserver() {
  for (map<void*, size_t>::iterator current = memoryUsers.begin(); current != memoryUsers.end(); current++) {
    memoryUsers.erase(current);
  }

  maximumSize = 0;
  totalSize = 0;
}

/**
 * Returns the singleton memory usage observer instance.
 *
 * \return memory usage observer instance
 */
MemoryUsageObserver* MemoryUsageObserver::getInstance() {
  if (instance == NULL) {
    instance = new MemoryUsageObserver;
  }

  return instance;
}

/**
 * Purges the current memory usage observer instance.
 */
void MemoryUsageObserver::purgeInstance() {
  if (instance != NULL) {
    delete instance;
  }

  instance = NULL;
}

/**
 * Adds memory.
 *
 * \param pointer to the allocated piece of memory
 * \param size of the allocated memory
 */
void MemoryUsageObserver::addMemory(void* pointer, size_t size) {
  // Memory might become reseized so we need to check whether the provided pointer is already tracked.
  map<void*, size_t>::iterator current = memoryUsers.find(pointer);
  if (current != memoryUsers.end()) {
    totalSize -= current->second;
  }

  memoryUsers[pointer] = size;
  totalSize += size;
  maximumSize = (totalSize > maximumSize) ? totalSize : maximumSize;
}

/**
 * Removes tracked memory. Prints a warning if untracked memory is to be released.
 *
 * \param pointer to the allocated piece of memory
 * \param *msg optional error message
 */
void MemoryUsageObserver::removeMemory(void* pointer, const char *msg) {
  map<void*, size_t>::iterator current = memoryUsers.find(pointer);

  if (current == memoryUsers.end()) {
    cout << "WARNING: There is non-tracked memory to be freed. Pointer "
      << pointer << " is not registered by MemoryUsageObserver: ";
    if (msg != NULL)
      cout << *msg;
    cout << endl;
    return;
  }

  totalSize -= current->second;
  memoryUsers.erase(current);
}

/**
 * Gets the size of currently allocated memory.
 */
size_t MemoryUsageObserver::getUsedMemorySize() {
  return totalSize;
}

/**
 * Gets the maximum size of allocated memory until now.
 */
size_t MemoryUsageObserver::getMaximumUsedMemory() {
  return maximumSize;
}

/**
 * Gets a map with pointers to the currently allocated memory ranges as keys and
 * the allocated size as value.
 */
map<void*, size_t> MemoryUsageObserver::getPointersToAllocatedMemory() {
  return memoryUsers;
}
