/** \file bond.cpp
 * 
 * Function implementations for the classes BondLeaf, BondTree and bond.
 * 
 */

#include "atom.hpp"
#include "bond.hpp"
#include "element.hpp"
#include "lists.hpp"


/***************************************** Functions for class bond ********************************/

/** Empty Constructor for class bond.
 */
bond::bond()
{
  leftatom = NULL;
  rightatom = NULL;
  previous = NULL;
  next = NULL;
  nr = -1;
  HydrogenBond = 0;
  BondDegree = 0;
  Used = white;
  Cyclic = false;
  Type = Undetermined;
};

/** Constructor for class bond, taking right and left bond partner
 * \param *left left atom
 * \param *right right atom
 * \param degree bond degree
 * \param number increasing index
 */
bond::bond(atom *left, atom *right, int degree=1, int number=0)
{
  leftatom = left;
  rightatom = right;
  previous = NULL;
  next = NULL;
  HydrogenBond = 0;
  if ((left != NULL) && (right != NULL)) {
    if ((left->type != NULL) && (left->type->Z == 1))
      HydrogenBond++;
    if ((right->type != NULL) && (right->type->Z == 1))
      HydrogenBond++;
  }
  BondDegree = degree;
  nr = number;
  Used = white;
  Cyclic = false;
};
bond::bond(atom *left, atom *right)
{
  leftatom = left;
  rightatom = right;
  previous = NULL;
  next = NULL;
  HydrogenBond = 0;
  if ((left != NULL) && (right != NULL)) {
    if ((left->type != NULL) && (left->type->Z == 1))
      HydrogenBond++;
    if ((right->type != NULL) && (right->type->Z == 1))
      HydrogenBond++;
  }
  BondDegree = 1;
  nr = 0;
  Used = white;
  Cyclic = false;
};

/** Empty Destructor for class bond.
 */
bond::~bond()
{
  // remove this node from the list structure
  if (previous != NULL) {
    previous->next = next;
  }
  if (next != NULL) {
    next->previous = previous;
  }
};

ostream & operator << (ostream &ost, const bond &b) 
{
  ost << "[" << b.leftatom->Name << " <" << b.BondDegree << "(H" << b.HydrogenBond << ")>" << b.rightatom->Name << "]";
  return ost;
};

/** Get the other atom in a bond if one is specified.
 * \param *Atom the pointer to the one atom
 * \return pointer to the other atom in the bond, NULL if no match (indicates something's wrong with the bond) 
 */
atom * bond::GetOtherAtom(atom *Atom) const
{
  if(leftatom == Atom) 
    return rightatom;
  if(rightatom == Atom) 
    return leftatom;
  cerr << "Bond " << *this << " does not contain atom " << *Atom << "!" << endl;
  return NULL;
};

/** Get the other atom in a bond if one is specified.
 * \param *Atom the pointer to the one atom
 * \return pointer to the other atom in the bond, NULL if no match (indicates something's wrong with the bond) 
 */
bond * bond::GetFirstBond()
{
  return GetFirst(this);
};

/** Get the other atom in a bond if one is specified.
 * \param *Atom the pointer to the one atom
 * \return pointer to the other atom in the bond, NULL if no match (indicates something's wrong with the bond) 
 */
bond * bond::GetLastBond()
{
  return GetLast(this);
};

/** Returns whether vertex was used in DFS.
 * \return bond::Used
 */
enum Shading bond::IsUsed() 
{
  return Used;
};

/** Checks if an atom exists in a bond.
 * \param *ptr pointer to atom
 * \return true if it is either bond::leftatom or bond::rightatom, false otherwise
 */
bool bond::Contains(const atom *ptr)
{
  return ((leftatom == ptr) || (rightatom == ptr));
};

/** Checks if an atom exists in a bond.
 * \param nr index of atom
 * \return true if it is either bond::leftatom or bond::rightatom, false otherwise
 */
bool bond::Contains(const int number)
{
  return ((leftatom->nr == number) || (rightatom->nr == number));
};

/** Masks vertex as used in DFS.
 * \return bond::Used, false if bond was already marked used
 */
bool bond::MarkUsed(enum Shading color) {
  if (Used == black) {
    cerr << "ERROR: Bond " << this << " was already marked black!." << endl;
    return false;
  } else {
    Used = color;
    return true;
  }
};

/** Resets used flag in DFS.
 * \return bond::Used
 */
void bond::ResetUsed() {
  Used = white;
};
