/*
 * \file PoolWorker.hpp
 *
 * This file strongly follows the Serialization example from the boost::asio
 * library (see client.cpp).
 *
 *  Created on: Feb 28, 2012
 *      Author: heber
 */

#ifndef POOLWORKER_HPP_
#define POOLWORKER_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <boost/asio.hpp>
#include <boost/function.hpp>
#include <vector>
#include "Connection.hpp"
#include "ExitflagContainer.hpp"
#include "Jobs/FragmentJob.hpp"
#include "Operations/Workers/EnrollInPoolOperation.hpp"
#include "Operations/Workers/RemoveFromPoolOperation.hpp"
#include "Operations/Workers/SubmitResultOperation.hpp"
#include "Listener.hpp"
#include "WorkerAddress.hpp"

/** Receives a job from Server to execute and return FragmentResult.
 *
 */
class PoolWorker : public ExitflagContainer
{
public:
  /// Constructor starts the asynchronous connect operation.
  PoolWorker(
      boost::asio::io_service& io_service,
      const std::string& host,
      const std::string& service,
      const std::string& listenhost,
      const std::string& listenservice);

  /** Returns the flag of the handled operation.
   *
   */
  size_t getFlag() const
  {
    if (PoolListener.getExitflag())
      return PoolListener.getExitflag();
    if (getExitflag())
      return getExitflag();
    return 0;
  }

  void WorkOnJob(FragmentJob::ptr &job);
  void removeFromPool();
  void shutdown(int sig);
  void shutdown();
  void finish();

  class PoolListener_t : public Listener
  {
  public:
    PoolListener_t(
        boost::asio::io_service& io_service,
        unsigned short port,
        PoolWorker &_callback) :
      Listener(io_service, port),
      callback(_callback)
    {}
    virtual ~PoolListener_t() {}

  protected:
    /// Handle completion of a accept controller operation.
    void handle_Accept(const boost::system::error_code& e, connection_ptr conn);

    /// Controller callback function when job has been sent.
    void handle_ReceiveJob(const boost::system::error_code& e, connection_ptr conn);

  private:
    //!> callback reference to PoolWorker for handling the job
    PoolWorker &callback;

    //!> current job
    FragmentJob::ptr job;
  };

private:
  //!> reference to io_service which we use for connections
  boost::asio::io_service& io_service;

  //!> The listener for the WorkerPool
  PoolListener_t PoolListener;

  //!> address of this worker
  const WorkerAddress address;

  //!> The Connection to the server for the stored operations
  Connection connection_;

  //!> internally bound function that sets the Exitflag to ErrorFlag
  boost::function<void ()> failed;

  //!> internally bound function that sets the Exitflag to ErrorFlag
  const boost::function<void ()> closingdown;

  //!> operation that handles obtaining a job
  EnrollInPoolOperation enrollOp;

  //!> operation that handles submitting job's result
  SubmitResultOperation submitOp;

  //!> internally bound function such that host and service don't have to be stored, submits result
  boost::function<void ()> submitresult;

  //!> operation that handles removal from pool
  RemoveFromPoolOperation removeOp;

  //!> internally bound function such that host and service don't have to be stored, removes us from server
  boost::function<void ()> removeme;
};

#endif /* POOLWORKER_HPP_ */
