/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2011 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * RemoveFromPoolOperation.cpp
 *
 *  Created on: Mar 01, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// boost asio needs specific operator new
#include <boost/asio.hpp>

#include "CodePatterns/MemDebug.hpp"

#include "RemoveFromPoolOperation.hpp"

#include <boost/bind.hpp>
#include <iostream>
#include "Connection.hpp" // Must come before boost/serialization headers.
#include "CodePatterns/Info.hpp"
#include "CodePatterns/Log.hpp"
#include "WorkerChoices.hpp"


/// Handle completion of a connect operation.
void RemoveFromPoolOperation::handle_connect(const boost::system::error_code& e,
    boost::asio::ip::tcp::resolver::iterator endpoint_iterator)
{
  Info info(__FUNCTION__);
  if (!e)
  {
    // Successfully established connection. Start operation to read the list
    // of jobs. The connection::async_write() function will automatically
    // decode the data that is read from the underlying socket.
    LOG(1, "INFO: Sending address " << address << " ...");
    connection_.async_write(address,
      boost::bind(&RemoveFromPoolOperation::handle_SendChoice, this,
      boost::asio::placeholders::error));
  } else if (endpoint_iterator != boost::asio::ip::tcp::resolver::iterator()) {
    // Try the next endpoint.
    connection_.socket().close();
    boost::asio::ip::tcp::endpoint endpoint = *endpoint_iterator;
    connection_.socket().async_connect(endpoint,
      boost::bind(&RemoveFromPoolOperation::handle_connect, this,
      boost::asio::placeholders::error, ++endpoint_iterator));
  } else {
    // An error occurred. Log it and return. Since we are not starting a new
    // operation the io_service will run out of work to do and the client will
    // exit.
    ELOG(1, e.message());
    AsyncOperation::handle_FinishOperation(e);
  }
}

/// Callback function when address has been sent and result is about to
void RemoveFromPoolOperation::handle_SendChoice(const boost::system::error_code& e)
{
  Info info(__FUNCTION__);
  if (!e)
  {
    // Successfully established connection. Start operation to read the list
    // of jobs. The connection::async_write() function will automatically
    // decode the data that is read from the underlying socket.
    enum WorkerChoices choice = RemoveFromPool;
    connection_.async_write(choice,
      boost::bind(&RemoveFromPoolOperation::handle_ReceiveFlag, this,
      boost::asio::placeholders::error));
  } else {
    // An error occurred. Log it and return. Since we are not starting a new
    // operation the io_service will run out of work to do and the client will
    // exit.
    ELOG(1, e.message());
    AsyncOperation::handle_FinishOperation(e);
  }
}

/// Callback function when address has been sent and result is about to
void RemoveFromPoolOperation::handle_ReceiveFlag(const boost::system::error_code& e)
{
  Info info(__FUNCTION__);
  if (!e)
  {
    // Successfully established connection. Start operation to read the list
    // of jobs. The connection::async_write() function will automatically
    // decode the data that is read from the underlying socket.
    connection_.async_read(RemovalFlag,
      boost::bind(&RemoveFromPoolOperation::handle_FinishOperation, this,
      boost::asio::placeholders::error));
  } else {
    // An error occurred. Log it and return. Since we are not starting a new
    // operation the io_service will run out of work to do and the client will
    // exit.
    ELOG(1, e.message());
    AsyncOperation::handle_FinishOperation(e);
  }
}

/// Callback function when result has been sent.
void RemoveFromPoolOperation::handle_FinishOperation(const boost::system::error_code& e)
{
  Info info(__FUNCTION__);

  if (RemovalFlag)
    LOG(1, "INFO: Removal successful.");
  else
    ELOG(1, "INFO: Removal failed.");

  AsyncOperation::handle_FinishOperation(e);
}
