/*
 * fast_functions.hpp
 *
 *  Created on: Apr 1, 2010
 *      Author: crueger
 */

#ifndef FAST_FUNCTIONS_HPP_
#define FAST_FUNCTIONS_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cmath>

#include "defs.hpp"

/**
 * !@file
 * This file contains several functions that need to be very fast and which are inlined for this
 * reason.
 *
 * Warning: do not forget inline keyword for functions in this file to avoid multiple definitions!
 */

typedef enum {
  Minus = -1,
  Zero = 0,
  Plus = +1
} sign_t;


/********************************* Functions ************************************************/

/** Gives numerical sign of some \a value.
 *
 * @param value to check
 * @return enumerated sign_t of value
 */
inline sign_t sign(double value)
{
  if(fabs(value)<LINALG_MYEPSILON()){
    return Zero;
  }
  if(value<0)
    return Minus;
  else
    return Plus;
}


/** hard-coded determinant of a 3x3 matrix.
 * \param a[9] matrix
 * \return \f$det(a)\f$
 */
inline double RDET3(const double a[NDIM*NDIM])
{
  return ((a)[0]*(a)[4]*(a)[8] + (a)[3]*(a)[7]*(a)[2] + (a)[6]*(a)[1]*(a)[5] - (a)[2]*(a)[4]*(a)[6] - (a)[5]*(a)[7]*(a)[0] - (a)[8]*(a)[1]*(a)[3]);
};

/** hard-coded determinant of a 2x2 matrix.
 * \param a[4] matrix
 * \return \f$det(a)\f$
 */
inline double RDET2(const double a[4])
{
  return ((a[0])*(a[3])-(a[1])*(a[2]));
};

/** hard-coded determinant of a 2x2 matrix.
 * \param a0 (0,0) entry of matrix
 * \param a1 (0,1) entry of matrix
 * \param a2 (1,0) entry of matrix
 * \param a3 (1,1) entry of matrix
 * \return \f$det(a)\f$
 */
inline double RDET2(const double a0, const double a1, const double a2, const double a3)
{
  return ((a0)*(a3)-(a1)*(a2));
};


#endif /* FAST_FUNCTIONS_HPP_ */
