#ifndef VECTOR_HPP_
#define VECTOR_HPP_

using namespace std;

/*********************************************** includes ***********************************/

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iosfwd>

#include <memory>
#include <vector>

#include "defs.hpp"
#include "Space.hpp"

/********************************************** declarations *******************************/

class Vector;
class RealSpaceMatrix;
class MatrixContent;
struct VectorContent;

typedef std::vector<Vector> pointset;

/** Single vector.
 * basically, just a x[3] but with helpful functions
 */
class Vector : public Space{
  friend Vector const operator*(const MatrixContent&,const Vector&);
  friend Vector const operator*(const Vector&,const MatrixContent&);
  friend class RealSpaceMatrix; // needs access to Vector(VectorContent*&) due to views
public:
  Vector();
  Vector(const double x1, const double x2, const double x3);
  Vector(const double x[3]);
  Vector(const Vector& src);
  virtual ~Vector();

  // Method implemented by forwarding to the Representation

  double DistanceSquared(const Vector &y) const;
  double DistanceToSpace(const Space& space) const;
  double ScalarProduct(const Vector &y) const;
  double Angle(const Vector &y) const;
  bool IsZero() const;
  bool IsOne() const;
  bool IsNormalTo(const Vector &normal) const;
  bool IsEqualTo(const Vector &a) const;

  void AddVector(const Vector &y);
  void SubtractVector(const Vector &y);
  void VectorProduct(const Vector &y);
  void ProjectOntoPlane(const Vector &y);
  void ProjectIt(const Vector &y);
  Vector Projection(const Vector &y) const;
  void ScaleAll(const double *factor);
  void ScaleAll(const Vector &factor);
  void Scale(const double factor);
  bool GetOneNormalVector(const Vector &x1);
  bool MakeNormalTo(const Vector &y1);
  std::pair<Vector,Vector> partition(const Vector&) const;
  std::pair<pointset,Vector> partition(const pointset&) const;
  size_t GreatestComponent() const;
  size_t SmallestComponent() const;

  // Accessors ussually come in pairs... and sometimes even more than that
  double& operator[](size_t i);
  const double& operator[](size_t i) const;
  double& at(size_t i);
  const double& at(size_t i) const;

  // Assignment operator
  Vector &operator=(const Vector& src);

  // Access to internal structure
  VectorContent* get() const;

  // Methods that are derived directly from other methods
  double Norm() const;
  double NormSquared() const;
  void Normalize();
  Vector getNormalized() const;
  void Zero();
  void One(const double one);
  void LinearCombinationOfVectors(const Vector &x1, const Vector &x2, const Vector &x3, const double * const factors);

  // operators for mathematical operations
  bool operator==(const Vector& b) const;
  bool operator!=(const Vector& b) const;
  const Vector& operator+=(const Vector& b);
  const Vector& operator-=(const Vector& b);
  Vector const operator+(const Vector& b) const;
  Vector const operator-(const Vector& b) const;
  const Vector& operator*=(const double m);
  const Vector operator*(const double m) const;

  // Methods inherited from Space
  virtual double distance(const Vector &point) const;
  virtual Vector getClosestPoint(const Vector &point) const;

  explicit Vector(VectorContent *&);
  explicit Vector(VectorContent &_content);
protected:

private:
  VectorContent *content;

};

// some commonly used and fixed vectors
const extern Vector zeroVec;
const extern Vector unitVec[NDIM];

ostream & operator << (ostream& ost, const Vector &m);
const Vector operator*(const double m, const Vector& a);

#endif /*VECTOR_HPP_*/
