/*
 * ShapeOps.cpp
 *
 *  Created on: Jun 18, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Helpers/MemDebug.hpp"

#include "Shapes/ShapeOps.hpp"
#include "Shapes/ShapeOps_impl.hpp"

#include "Helpers/Assert.hpp"

/********************* Resize ********************/

Resize_impl::Resize_impl(const Shape::impl_ptr &_arg,double _size) :
  arg(_arg), size(_size)
{
  ASSERT(size>0,"Cannot resize a Shape to size zero or below");
}

Resize_impl::~Resize_impl(){}

bool Resize_impl::isInside(const Vector& point){
  return arg->isInside((1/size) * point);
}

Shape resize(const Shape &arg,double size){
  Shape::impl_ptr impl = Shape::impl_ptr(new Resize_impl(getShapeImpl(arg),size));
  return Shape(impl);
}

/*************************** translate *******************/

Translate_impl::Translate_impl(const Shape::impl_ptr &_arg, const Vector &_offset) :
  arg(_arg),offset(_offset)
{}

Translate_impl::~Translate_impl(){}

bool Translate_impl::isInside(const Vector& point){
  return arg->isInside(point-offset);
}

Shape translate(const Shape &arg, const Vector &offset){
  Shape::impl_ptr impl = Shape::impl_ptr(new Translate_impl(getShapeImpl(arg),offset));
  return Shape(impl);
}

/*********************** stretch ******************/

Stretch_impl::Stretch_impl(const Shape::impl_ptr &_arg, const Vector &_factors) :
  arg(_arg),factors(_factors)
{
  ASSERT(factors[0]>0,"cannot stretch a shape by a negative amount");
  ASSERT(factors[1]>0,"cannot stretch a shape by a negative amount");
  ASSERT(factors[2]>0,"cannot stretch a shape by a negative amount");
  for(int i = NDIM;i--;){
    reciFactors[i] = 1/factors[i];
  }
}

Stretch_impl::~Stretch_impl(){}

bool Stretch_impl::isInside(const Vector& point){
  Vector helper=point;
  helper.ScaleAll(reciFactors);
  return arg->isInside(helper);
}

Shape stretch(const Shape &arg, const Vector &factors){
  Shape::impl_ptr impl = Shape::impl_ptr(new Stretch_impl(getShapeImpl(arg),factors));
  return Shape(impl);
}

/************************* transform *****************/

Transform_impl::Transform_impl(const Shape::impl_ptr &_arg, const Matrix &_transformation) :
  arg(_arg),transformation(_transformation)
{
  transformationInv = transformation.invert();
}

Transform_impl::~Transform_impl(){}

bool Transform_impl::isInside(const Vector& point){
  return arg->isInside(transformationInv * point);
}

Shape transform(const Shape &arg, const Matrix &transformation){
  Shape::impl_ptr impl = Shape::impl_ptr(new Transform_impl(getShapeImpl(arg),transformation));
  return Shape(impl);
}
