/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * RotateAroundSelfByAngleAction.cpp
 *
 *  Created on: Aug 06, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Helpers/MemDebug.hpp"

#include "Actions/MoleculeAction/RotateAroundSelfByAngleAction.hpp"
#include "Actions/ActionRegistry.hpp"
#include "Helpers/Log.hpp"
#include "Helpers/Verbose.hpp"
#include "LinearAlgebra/Line.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "atom.hpp"
#include "molecule.hpp"


#include <iostream>
#include <fstream>
#include <string>

using namespace std;

#include "UIElements/UIFactory.hpp"
#include "UIElements/Dialog.hpp"
#include "Actions/ValueStorage.hpp"

/****** MoleculeRotateAroundSelfByAngleAction *****/

// memento to remember the state when undoing

class MoleculeRotateAroundSelfByAngleState : public ActionState {
public:
    MoleculeRotateAroundSelfByAngleState(molecule * const _mol, const Vector &_Axis, const double _alpha) :
      mol(_mol),
      Axis(_Axis),
      alpha(_alpha)
    {}
    molecule* const mol;
    Vector Axis;
    double alpha;
};

const char MoleculeRotateAroundSelfByAngleAction::NAME[] = "rotate-self";

MoleculeRotateAroundSelfByAngleAction::MoleculeRotateAroundSelfByAngleAction() :
  Action(NAME)
{}

MoleculeRotateAroundSelfByAngleAction::~MoleculeRotateAroundSelfByAngleAction()
{}

void MoleculeRotateAroundSelfByAngle(const Vector &Axis, const double angle) {
  ValueStorage::getInstance().setCurrentValue(MoleculeRotateAroundSelfByAngleAction::NAME, angle);
  ValueStorage::getInstance().setCurrentValue("position", Axis);
  ActionRegistry::getInstance().getActionByName(MoleculeRotateAroundSelfByAngleAction::NAME)->call(Action::NonInteractive);
};

Dialog* MoleculeRotateAroundSelfByAngleAction::fillDialog(Dialog *dialog) {
  ASSERT(dialog,"No Dialog given when filling action dialog");

  dialog->queryDouble(NAME, ValueStorage::getInstance().getDescription(NAME));
  dialog->queryVector("position", false, ValueStorage::getInstance().getDescription(NAME));

  return dialog;
}

Action::state_ptr MoleculeRotateAroundSelfByAngleAction::performCall() {
  molecule *mol = NULL;
  double alpha = 0.;
  Vector Axis;

  // obtain angle and axis around which rotate
  ValueStorage::getInstance().queryCurrentValue(NAME, alpha);
  ValueStorage::getInstance().queryCurrentValue("position", Axis);

  // check whether a single atom and molecule is selected
  if (World::getInstance().getSelectedMolecules().size() != 1)
    return Action::failure;
  mol = World::getInstance().beginMoleculeSelection()->second;

  // check whether Axis is valid
  if (Axis.IsZero())
    return Action::failure;

  // convert from degrees to radian
  alpha *= M_PI/180.;

  // Creation Line that is the rotation axis
  Vector *CenterOfGravity = mol->DetermineCenterOfGravity();
  Line RotationAxis(*CenterOfGravity, Axis);
  delete(CenterOfGravity);
  DoLog(0) && (Log() << Verbose(0) << "Rotate around self by " << alpha << " along " << RotationAxis << "." << endl);

  for (molecule::iterator iter = mol->begin(); iter != mol->end(); ++iter) {
    (*iter)->setPosition(RotationAxis.rotateVector((*iter)->getPosition(), alpha));
  }
  DoLog(0) && (Log() << Verbose(0) << "done." << endl);

  return Action::state_ptr(new MoleculeRotateAroundSelfByAngleState(mol, Axis, alpha));
}

Action::state_ptr MoleculeRotateAroundSelfByAngleAction::performUndo(Action::state_ptr _state) {
  MoleculeRotateAroundSelfByAngleState *state = assert_cast<MoleculeRotateAroundSelfByAngleState*>(_state.get());

  Vector *CenterOfGravity = state->mol->DetermineCenterOfGravity();
  Line RotationAxis(*CenterOfGravity, state->Axis);
  delete(CenterOfGravity);

  for (molecule::iterator iter = state->mol->begin(); iter != state->mol->end(); ++iter) {
    (*iter)->setPosition(RotationAxis.rotateVector((*iter)->getPosition(), -state->alpha));
  }

  return Action::state_ptr(_state);
}

Action::state_ptr MoleculeRotateAroundSelfByAngleAction::performRedo(Action::state_ptr _state){
  MoleculeRotateAroundSelfByAngleState *state = assert_cast<MoleculeRotateAroundSelfByAngleState*>(_state.get());

  Vector *CenterOfGravity = state->mol->DetermineCenterOfGravity();
  Line RotationAxis(*CenterOfGravity, state->Axis);
  delete(CenterOfGravity);

  for (molecule::iterator iter = state->mol->begin(); iter != state->mol->end(); ++iter) {
    (*iter)->setPosition(RotationAxis.rotateVector((*iter)->getPosition(), state->alpha));
  }

  return Action::state_ptr(_state);
}

bool MoleculeRotateAroundSelfByAngleAction::canUndo() {
  return true;
}

bool MoleculeRotateAroundSelfByAngleAction::shouldUndo() {
  return true;
}

const string MoleculeRotateAroundSelfByAngleAction::getName() {
  return NAME;
}
