/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * QtTimeLine.cpp
 *
 *  Created on: Jul 11, 2013
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// make sure Qt includes come before MemDebug
#include "QtTimeLine.hpp"

#include "CodePatterns/MemDebug.hpp"

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"

#include "Actions/WorldAction/SetWorldTimeAction.hpp"
#include "Atom/atom.hpp"
#include "Atom/AtomObserver.hpp"
#include "WorldTime.hpp"

QtTimeLine::QtTimeLine(QWidget * _parent) :
    QSlider (Qt::Horizontal, _parent),
    Observer("QtTimeLine"),
    atomobserver_enlisted(false),
    worldtime_enlisted(false),
    WorldTime_change(false)
{
  // set initial values
  QSlider::setMinimum(0);
  QSlider::setMaximum(0);
  WorldTime_change = true;
  QSlider::setValue(0);
  QSlider::setTickInterval(1);
  QSlider::setSingleStep(1);
  QSlider::setTracking(true);

  // sign on to observable
  WorldTime::getInstance().signOn(this, WorldTime::TimeChanged);
  worldtime_enlisted = true;
  AtomObserver::getInstance().signOn(this, AtomObservable::TrajectoryChanged);
  atomobserver_enlisted = true;

  // connect to QSlider's valueChanged() signal
  bool result = connect(this,SIGNAL(valueChanged(int)),this,SLOT(StepUpdate(int)), Qt::DirectConnection);
  if (!result)
    ELOG(0, "Could not connect to QSlider::valueChanged.");
}

QtTimeLine::~QtTimeLine()
{
  if (worldtime_enlisted)
    WorldTime::getInstance().signOff(this, WorldTime::TimeChanged);
  if (atomobserver_enlisted)
    AtomObserver::getInstance().signOff(this, AtomObservable::TrajectoryChanged);
}

void QtTimeLine::subjectKilled(Observable *publisher)
{
  // as long as we are enlisted to only one channel, we have nothing to worry.
  // Otherwise we should also signOn to global changes for subjectKilled() and
  // sign off from the channels
  if (static_cast<AtomObserver *>(publisher) == AtomObserver::getPointer()) {
    atomobserver_enlisted = false;
  } else if (static_cast<WorldTime *>(publisher) == WorldTime::getPointer()) {
    worldtime_enlisted = false;
 }
}

void QtTimeLine::update(Observable *publisher)
{
  ELOG(0, "We are not enlisted for general updates.");
}

void QtTimeLine::recieveNotification(Observable *publisher, Notification_ptr notification)
{
  if (static_cast<WorldTime *>(publisher) == WorldTime::getPointer()) {
    const int timestep = WorldTime::getTime();
    // check whether we are beyond maximum
    if (timestep > QSlider::maximum())
      QSlider::setMaximum(timestep);
    // set slider position to new time step
    WorldTime_change = true;
    QSlider::setValue( timestep );
  } else
    //if (static_cast<AtomObserver *>(publisher) == AtomObserver::getPointer())
  {
    // calculate max trajectory (need dynamic_cast due to virtual base)
    const atom *_atom = dynamic_cast<const atom *>(publisher);
    const int MaxTrajectory = _atom->getTrajectorySize()-1;
    if (MaxTrajectory > QSlider::maximum())
      QSlider::setMaximum(MaxTrajectory);
  }
}

void QtTimeLine::StepUpdate(int position)
{
  if (WorldTime_change)
    WorldTime_change = false;
  else
    MoleCuilder::WorldSetWorldTime(position);
}
