/** \file MoleculeListClass.cpp
 *
 * Function implementations for the class MoleculeListClass.
 *
 */

#include <cstring>

#include "atom.hpp"
#include "bond.hpp"
#include "boundary.hpp"
#include "config.hpp"
#include "element.hpp"
#include "helpers.hpp"
#include "linkedcell.hpp"
#include "lists.hpp"
#include "log.hpp"
#include "molecule.hpp"
#include "memoryallocator.hpp"
#include "periodentafel.hpp"

/*********************************** Functions for class MoleculeListClass *************************/

/** Constructor for MoleculeListClass.
 */
MoleculeListClass::MoleculeListClass()
{
  // empty lists
  ListOfMolecules.clear();
  MaxIndex = 1;
};

/** Destructor for MoleculeListClass.
 */
MoleculeListClass::~MoleculeListClass()
{
  Log() << Verbose(3) << this << ": Freeing ListOfMolcules." << endl;
  for (MoleculeList::iterator ListRunner = ListOfMolecules.begin(); ListRunner != ListOfMolecules.end(); ListRunner++) {
    Log() << Verbose(4) << "ListOfMolecules: Freeing " << *ListRunner << "." << endl;
    delete (*ListRunner);
  }
  Log() << Verbose(4) << "Freeing ListOfMolecules." << endl;
  ListOfMolecules.clear(); // empty list
};

/** Insert a new molecule into the list and set its number.
 * \param *mol molecule to add to list.
 * \return true - add successful
 */
void MoleculeListClass::insert(molecule *mol)
{
  mol->IndexNr = MaxIndex++;
  ListOfMolecules.push_back(mol);
};

/** Compare whether two molecules are equal.
 * \param *a molecule one
 * \param *n molecule two
 * \return lexical value (-1, 0, +1)
 */
int MolCompare(const void *a, const void *b)
{
  int *aList = NULL, *bList = NULL;
  int Count, Counter, aCounter, bCounter;
  int flag;
  atom *aWalker = NULL;
  atom *bWalker = NULL;

  // sort each atom list and put the numbers into a list, then go through
  //Log() << Verbose(0) << "Comparing fragment no. " << *(molecule **)a << " to " << *(molecule **)b << "." << endl;
  if ((**(molecule **) a).AtomCount < (**(molecule **) b).AtomCount) {
    return -1;
  } else {
    if ((**(molecule **) a).AtomCount > (**(molecule **) b).AtomCount)
      return +1;
    else {
      Count = (**(molecule **) a).AtomCount;
      aList = new int[Count];
      bList = new int[Count];

      // fill the lists
      aWalker = (**(molecule **) a).start;
      bWalker = (**(molecule **) b).start;
      Counter = 0;
      aCounter = 0;
      bCounter = 0;
      while ((aWalker->next != (**(molecule **) a).end) && (bWalker->next != (**(molecule **) b).end)) {
        aWalker = aWalker->next;
        bWalker = bWalker->next;
        if (aWalker->GetTrueFather() == NULL)
          aList[Counter] = Count + (aCounter++);
        else
          aList[Counter] = aWalker->GetTrueFather()->nr;
        if (bWalker->GetTrueFather() == NULL)
          bList[Counter] = Count + (bCounter++);
        else
          bList[Counter] = bWalker->GetTrueFather()->nr;
        Counter++;
      }
      // check if AtomCount was for real
      flag = 0;
      if ((aWalker->next == (**(molecule **) a).end) && (bWalker->next != (**(molecule **) b).end)) {
        flag = -1;
      } else {
        if ((aWalker->next != (**(molecule **) a).end) && (bWalker->next == (**(molecule **) b).end))
          flag = 1;
      }
      if (flag == 0) {
        // sort the lists
        gsl_heapsort(aList, Count, sizeof(int), CompareDoubles);
        gsl_heapsort(bList, Count, sizeof(int), CompareDoubles);
        // compare the lists

        flag = 0;
        for (int i = 0; i < Count; i++) {
          if (aList[i] < bList[i]) {
            flag = -1;
          } else {
            if (aList[i] > bList[i])
              flag = 1;
          }
          if (flag != 0)
            break;
        }
      }
      delete[] (aList);
      delete[] (bList);
      return flag;
    }
  }
  return -1;
};

/** Output of a list of all molecules.
 * \param *out output stream
 */
void MoleculeListClass::Enumerate(ofstream *out)
{
  element* Elemental = NULL;
  atom *Walker = NULL;
  int Counts[MAX_ELEMENTS];
  double size=0;
  Vector Origin;

  // header
  Log() << Verbose(0) << "Index\tName\t\tAtoms\tFormula\tCenter\tSize" << endl;
  Log() << Verbose(0) << "-----------------------------------------------" << endl;
  if (ListOfMolecules.size() == 0)
    Log() << Verbose(0) << "\tNone" << endl;
  else {
    Origin.Zero();
    for (MoleculeList::iterator ListRunner = ListOfMolecules.begin(); ListRunner != ListOfMolecules.end(); ListRunner++) {
      // reset element counts
      for (int j = 0; j<MAX_ELEMENTS;j++)
        Counts[j] = 0;
      // count atoms per element and determine size of bounding sphere
      size=0.;
      Walker = (*ListRunner)->start;
      while (Walker->next != (*ListRunner)->end) {
        Walker = Walker->next;
        Counts[Walker->type->Z]++;
        if (Walker->x.DistanceSquared(&Origin) > size)
          size = Walker->x.DistanceSquared(&Origin);
      }
      // output Index, Name, number of atoms, chemical formula
      Log() << Verbose(0) << ((*ListRunner)->ActiveFlag ? "*" : " ") << (*ListRunner)->IndexNr << "\t" << (*ListRunner)->name << "\t\t" << (*ListRunner)->AtomCount << "\t";
      Elemental = (*ListRunner)->elemente->end;
      while(Elemental->previous != (*ListRunner)->elemente->start) {
        Elemental = Elemental->previous;
        if (Counts[Elemental->Z] != 0)
          Log() << Verbose(0) << Elemental->symbol << Counts[Elemental->Z];
      }
      // Center and size
      Log() << Verbose(0) << "\t" << (*ListRunner)->Center << "\t" << sqrt(size) << endl;
    }
  }
};

/** Returns the molecule with the given index \a index.
 * \param index index of the desired molecule
 * \return pointer to molecule structure, NULL if not found
 */
molecule * MoleculeListClass::ReturnIndex(int index)
{
  for(MoleculeList::iterator ListRunner = ListOfMolecules.begin(); ListRunner != ListOfMolecules.end(); ListRunner++)
    if ((*ListRunner)->IndexNr == index)
      return (*ListRunner);
  return NULL;
};

/** Simple merge of two molecules into one.
 * \param *mol destination molecule
 * \param *srcmol source molecule
 * \return true - merge successful, false - merge failed (probably due to non-existant indices
 */
bool MoleculeListClass::SimpleMerge(molecule *mol, molecule *srcmol)
{
  if (srcmol == NULL)
    return false;

  // put all molecules of src into mol
  atom *Walker = srcmol->start;
  atom *NextAtom = Walker->next;
  while (NextAtom != srcmol->end) {
    Walker = NextAtom;
    NextAtom = Walker->next;
    srcmol->UnlinkAtom(Walker);
    mol->AddAtom(Walker);
  }

  // remove src
  ListOfMolecules.remove(srcmol);
  delete(srcmol);
  return true;
};

/** Simple add of one molecules into another.
 * \param *mol destination molecule
 * \param *srcmol source molecule
 * \return true - merge successful, false - merge failed (probably due to non-existant indices
 */
bool MoleculeListClass::SimpleAdd(molecule *mol, molecule *srcmol)
{
  if (srcmol == NULL)
    return false;

  // put all molecules of src into mol
  atom *Walker = srcmol->start;
  atom *NextAtom = Walker->next;
  while (NextAtom != srcmol->end) {
    Walker = NextAtom;
    NextAtom = Walker->next;
    Walker = mol->AddCopyAtom(Walker);
    Walker->father = Walker;
  }

  return true;
};

/** Simple merge of a given set of molecules into one.
 * \param *mol destination molecule
 * \param *src index of set of source molecule
 * \param N number of source molecules
 * \return true - merge successful, false - some merges failed (probably due to non-existant indices)
 */
bool MoleculeListClass::SimpleMultiMerge(molecule *mol, int *src, int N)
{
  bool status = true;
  // check presence of all source molecules
  for (int i=0;i<N;i++) {
    molecule *srcmol = ReturnIndex(src[i]);
    status = status && SimpleMerge(mol, srcmol);
  }
  return status;
};

/** Simple add of a given set of molecules into one.
 * \param *mol destination molecule
 * \param *src index of set of source molecule
 * \param N number of source molecules
 * \return true - merge successful, false - some merges failed (probably due to non-existant indices)
 */
bool MoleculeListClass::SimpleMultiAdd(molecule *mol, int *src, int N)
{
  bool status = true;
  // check presence of all source molecules
  for (int i=0;i<N;i++) {
    molecule *srcmol = ReturnIndex(src[i]);
    status = status && SimpleAdd(mol, srcmol);
  }
  return status;
};

/** Scatter merge of a given set of molecules into one.
 * Scatter merge distributes the molecules in such a manner that they don't overlap.
 * \param *mol destination molecule
 * \param *src index of set of source molecule
 * \param N number of source molecules
 * \return true - merge successful, false - merge failed (probably due to non-existant indices
 * \TODO find scatter center for each src molecule
 */
bool MoleculeListClass::ScatterMerge(molecule *mol, int *src, int N)
{
  // check presence of all source molecules
  for (int i=0;i<N;i++) {
    // get pointer to src molecule
    molecule *srcmol = ReturnIndex(src[i]);
    if (srcmol == NULL)
      return false;
  }
  // adapt each Center
  for (int i=0;i<N;i++) {
    // get pointer to src molecule
    molecule *srcmol = ReturnIndex(src[i]);
    //srcmol->Center.Zero();
    srcmol->Translate(&srcmol->Center);
  }
  // perform a simple multi merge
  SimpleMultiMerge(mol, src, N);
  return true;
};

/** Embedding merge of a given set of molecules into one.
 * Embedding merge inserts one molecule into the other.
 * \param *mol destination molecule (fixed one)
 * \param *srcmol source molecule (variable one, where atoms are taken from)
 * \return true - merge successful, false - merge failed (probably due to non-existant indices)
 * \TODO linked cell dimensions for boundary points has to be as big as inner diameter!
 */
bool MoleculeListClass::EmbedMerge(molecule *mol, molecule *srcmol)
{
  LinkedCell *LCList = NULL;
  Tesselation *TesselStruct = NULL;
  if ((srcmol == NULL) || (mol == NULL)) {
    eLog() << Verbose(1) << "Either fixed or variable molecule is given as NULL." << endl;
    return false;
  }

  // calculate envelope for *mol
  LCList = new LinkedCell(mol, 8.);
  FindNonConvexBorder(mol, TesselStruct, (const LinkedCell *&)LCList, 4., NULL);
  if (TesselStruct == NULL) {
    eLog() << Verbose(1) << "Could not tesselate the fixed molecule." << endl;
    return false;
  }
  delete(LCList);
  LCList = new LinkedCell(TesselStruct, 8.);  // re-create with boundary points only!

  // prepare index list for bonds
  srcmol->CountAtoms();
  atom ** CopyAtoms = new atom*[srcmol->AtomCount];
  for(int i=0;i<srcmol->AtomCount;i++)
    CopyAtoms[i] = NULL;

  // for each of the source atoms check whether we are in- or outside and add copy atom
  atom *Walker = srcmol->start;
  int nr=0;
  while (Walker->next != srcmol->end) {
    Walker = Walker->next;
    Log() << Verbose(2) << "INFO: Current Walker is " << *Walker << "." << endl;
    if (!TesselStruct->IsInnerPoint(Walker->x, LCList)) {
      CopyAtoms[Walker->nr] = new atom(Walker);
      mol->AddAtom(CopyAtoms[Walker->nr]);
      nr++;
    } else {
      // do nothing
    }
  }
  Log() << Verbose(1) << nr << " of " << srcmol->AtomCount << " atoms have been merged.";

  // go through all bonds and add as well
  bond *Binder = srcmol->first;
  while(Binder->next != srcmol->last) {
    Binder = Binder->next;
    Log() << Verbose(3) << "Adding Bond between " << *CopyAtoms[Binder->leftatom->nr] << " and " << *CopyAtoms[Binder->rightatom->nr]<< "." << endl;
    mol->AddBond(CopyAtoms[Binder->leftatom->nr], CopyAtoms[Binder->rightatom->nr], Binder->BondDegree);
  }
  delete(LCList);
  return true;
};

/** Simple output of the pointers in ListOfMolecules.
 * \param *out output stream
 */
void MoleculeListClass::Output(ofstream *out)
{
  Log() << Verbose(1) << "MoleculeList: ";
  for (MoleculeList::iterator ListRunner = ListOfMolecules.begin(); ListRunner != ListOfMolecules.end(); ListRunner++)
    Log() << Verbose(0) << *ListRunner << "\t";
  Log() << Verbose(0) << endl;
};

/** Calculates necessary hydrogen correction due to unwanted interaction between saturated ones.
 * If for a pair of two hydrogen atoms a and b, at least is a saturated one, and a and b are not
 * bonded to the same atom, then we add for this pair a correction term constructed from a Morse
 * potential function fit to QM calculations with respecting to the interatomic hydrogen distance.
 * \param *out output stream for debugging
 * \param *path path to file
 */
bool MoleculeListClass::AddHydrogenCorrection(char *path)
{
  atom *Walker = NULL;
  atom *Runner = NULL;
  bond *Binder = NULL;
  double ***FitConstant = NULL, **correction = NULL;
  int a, b;
  ofstream output;
  ifstream input;
  string line;
  stringstream zeile;
  double distance;
  char ParsedLine[1023];
  double tmp;
  char *FragmentNumber = NULL;

  Log() << Verbose(1) << "Saving hydrogen saturation correction ... ";
  // 0. parse in fit constant files that should have the same dimension as the final energy files
  // 0a. find dimension of matrices with constants
  line = path;
  line.append("/");
  line += FRAGMENTPREFIX;
  line += "1";
  line += FITCONSTANTSUFFIX;
  input.open(line.c_str());
  if (input == NULL) {
    Log() << Verbose(1) << endl << "Unable to open " << line << ", is the directory correct?" << endl;
    return false;
  }
  a = 0;
  b = -1; // we overcount by one
  while (!input.eof()) {
    input.getline(ParsedLine, 1023);
    zeile.str(ParsedLine);
    int i = 0;
    while (!zeile.eof()) {
      zeile >> distance;
      i++;
    }
    if (i > a)
      a = i;
    b++;
  }
  Log() << Verbose(0) << "I recognized " << a << " columns and " << b << " rows, ";
  input.close();

  // 0b. allocate memory for constants
  FitConstant = Calloc<double**>(3, "MoleculeListClass::AddHydrogenCorrection: ***FitConstant");
  for (int k = 0; k < 3; k++) {
    FitConstant[k] = Calloc<double*>(a, "MoleculeListClass::AddHydrogenCorrection: **FitConstant[]");
    for (int i = a; i--;) {
      FitConstant[k][i] = Calloc<double>(b, "MoleculeListClass::AddHydrogenCorrection: *FitConstant[][]");
    }
  }
  // 0c. parse in constants
  for (int i = 0; i < 3; i++) {
    line = path;
    line.append("/");
    line += FRAGMENTPREFIX;
    sprintf(ParsedLine, "%d", i + 1);
    line += ParsedLine;
    line += FITCONSTANTSUFFIX;
    input.open(line.c_str());
    if (input == NULL) {
      eLog() << Verbose(0) << endl << "Unable to open " << line << ", is the directory correct?" << endl;
      performCriticalExit();
      return false;
    }
    int k = 0, l;
    while ((!input.eof()) && (k < b)) {
      input.getline(ParsedLine, 1023);
      //Log() << Verbose(0) << "Current Line: " << ParsedLine << endl;
      zeile.str(ParsedLine);
      zeile.clear();
      l = 0;
      while ((!zeile.eof()) && (l < a)) {
        zeile >> FitConstant[i][l][k];
        //Log() << Verbose(0) << FitConstant[i][l][k] << "\t";
        l++;
      }
      //Log() << Verbose(0) << endl;
      k++;
    }
    input.close();
  }
  for (int k = 0; k < 3; k++) {
    Log() << Verbose(0) << "Constants " << k << ":" << endl;
    for (int j = 0; j < b; j++) {
      for (int i = 0; i < a; i++) {
        Log() << Verbose(0) << FitConstant[k][i][j] << "\t";
      }
      Log() << Verbose(0) << endl;
    }
    Log() << Verbose(0) << endl;
  }

  // 0d. allocate final correction matrix
  correction = Calloc<double*>(a, "MoleculeListClass::AddHydrogenCorrection: **correction");
  for (int i = a; i--;)
    correction[i] = Calloc<double>(b, "MoleculeListClass::AddHydrogenCorrection: *correction[]");

  // 1a. go through every molecule in the list
  for (MoleculeList::iterator ListRunner = ListOfMolecules.begin(); ListRunner != ListOfMolecules.end(); ListRunner++) {
    // 1b. zero final correction matrix
    for (int k = a; k--;)
      for (int j = b; j--;)
        correction[k][j] = 0.;
    // 2. take every hydrogen that is a saturated one
    Walker = (*ListRunner)->start;
    while (Walker->next != (*ListRunner)->end) {
      Walker = Walker->next;
      //Log() << Verbose(1) << "Walker: " << *Walker << " with first bond " << *(Walker->ListOfBonds.begin()) << "." << endl;
      if ((Walker->type->Z == 1) && ((Walker->father == NULL)
          || (Walker->father->type->Z != 1))) { // if it's a hydrogen
        Runner = (*ListRunner)->start;
        while (Runner->next != (*ListRunner)->end) {
          Runner = Runner->next;
          //Log() << Verbose(2) << "Runner: " << *Runner << " with first bond " << *(Walker->ListOfBonds.begin()) << "." << endl;
          // 3. take every other hydrogen that is the not the first and not bound to same bonding partner
          Binder = *(Runner->ListOfBonds.begin());
          if ((Runner->type->Z == 1) && (Runner->nr > Walker->nr) && (Binder->GetOtherAtom(Runner) != Binder->GetOtherAtom(Walker))) { // (hydrogens have only one bonding partner!)
            // 4. evaluate the morse potential for each matrix component and add up
            distance = Runner->x.Distance(&Walker->x);
            //Log() << Verbose(0) << "Fragment " << (*ListRunner)->name << ": " << *Runner << "<= " << distance << "=>" << *Walker << ":" << endl;
            for (int k = 0; k < a; k++) {
              for (int j = 0; j < b; j++) {
                switch (k) {
                  case 1:
                  case 7:
                  case 11:
                    tmp = pow(FitConstant[0][k][j] * (1. - exp(-FitConstant[1][k][j] * (distance - FitConstant[2][k][j]))), 2);
                    break;
                  default:
                    tmp = FitConstant[0][k][j] * pow(distance, FitConstant[1][k][j]) + FitConstant[2][k][j];
                };
                correction[k][j] -= tmp; // ground state is actually lower (disturbed by additional interaction)
                //Log() << Verbose(0) << tmp << "\t";
              }
              //Log() << Verbose(0) << endl;
            }
            //Log() << Verbose(0) << endl;
          }
        }
      }
    }
    // 5. write final matrix to file
    line = path;
    line.append("/");
    line += FRAGMENTPREFIX;
    FragmentNumber = FixedDigitNumber(ListOfMolecules.size(), (*ListRunner)->IndexNr);
    line += FragmentNumber;
    delete (FragmentNumber);
    line += HCORRECTIONSUFFIX;
    output.open(line.c_str());
    output << "Time\t\tTotal\t\tKinetic\t\tNonLocal\tCorrelation\tExchange\tPseudo\t\tHartree\t\t-Gauss\t\tEwald\t\tIonKin\t\tETotal" << endl;
    for (int j = 0; j < b; j++) {
      for (int i = 0; i < a; i++)
        output << correction[i][j] << "\t";
      output << endl;
    }
    output.close();
  }
  line = path;
  line.append("/");
  line += HCORRECTIONSUFFIX;
  output.open(line.c_str());
  output << "Time\t\tTotal\t\tKinetic\t\tNonLocal\tCorrelation\tExchange\tPseudo\t\tHartree\t\t-Gauss\t\tEwald\t\tIonKin\t\tETotal" << endl;
  for (int j = 0; j < b; j++) {
    for (int i = 0; i < a; i++)
      output << 0 << "\t";
    output << endl;
  }
  output.close();
  // 6. free memory of parsed matrices
  for (int k = 0; k < 3; k++) {
    for (int i = a; i--;) {
      Free(&FitConstant[k][i]);
    }
    Free(&FitConstant[k]);
  }
  Free(&FitConstant);
  Log() << Verbose(0) << "done." << endl;
  return true;
};

/** Store force indices, i.e. the connection between the nuclear index in the total molecule config and the respective atom in fragment config.
 * \param *out output stream for debugging
 * \param *path path to file
 * \param *SortIndex Index to map from the BFS labeling to the sequence how of Ion_Type in the config
 * \return true - file written successfully, false - writing failed
 */
bool MoleculeListClass::StoreForcesFile(char *path,
    int *SortIndex)
{
  bool status = true;
  ofstream ForcesFile;
  stringstream line;
  atom *Walker = NULL;
  element *runner = NULL;

  // open file for the force factors
  Log() << Verbose(1) << "Saving  force factors ... ";
  line << path << "/" << FRAGMENTPREFIX << FORCESFILE;
  ForcesFile.open(line.str().c_str(), ios::out);
  if (ForcesFile != NULL) {
    //Log() << Verbose(1) << "Final AtomicForcesList: ";
    //output << prefix << "Forces" << endl;
    for (MoleculeList::iterator ListRunner = ListOfMolecules.begin(); ListRunner != ListOfMolecules.end(); ListRunner++) {
      runner = (*ListRunner)->elemente->start;
      while (runner->next != (*ListRunner)->elemente->end) { // go through every element
        runner = runner->next;
        if ((*ListRunner)->ElementsInMolecule[runner->Z]) { // if this element got atoms
          Walker = (*ListRunner)->start;
          while (Walker->next != (*ListRunner)->end) { // go through every atom of this element
            Walker = Walker->next;
            if (Walker->type->Z == runner->Z) {
              if ((Walker->GetTrueFather() != NULL) && (Walker->GetTrueFather() != Walker)) {// if there is a rea
                //Log() << Verbose(0) << "Walker is " << *Walker << " with true father " << *( Walker->GetTrueFather()) << ", it
                ForcesFile << SortIndex[Walker->GetTrueFather()->nr] << "\t";
              } else
                // otherwise a -1 to indicate an added saturation hydrogen
                ForcesFile << "-1\t";
            }
          }
        }
      }
      ForcesFile << endl;
    }
    ForcesFile.close();
    Log() << Verbose(1) << "done." << endl;
  } else {
    status = false;
    Log() << Verbose(1) << "failed to open file " << line.str() << "." << endl;
  }
  ForcesFile.close();

  return status;
};

/** Writes a config file for each molecule in the given \a **FragmentList.
 * \param *out output stream for debugging
 * \param *configuration standard configuration to attach atoms in fragment molecule to.
 * \param *SortIndex Index to map from the BFS labeling to the sequence how of Ion_Type in the config
 * \param DoPeriodic true - call ScanForPeriodicCorrection, false - don't
 * \param DoCentering true - call molecule::CenterEdge(), false - don't
 * \return true - success (each file was written), false - something went wrong.
 */
bool MoleculeListClass::OutputConfigForListOfFragments(config *configuration, int *SortIndex)
{
  ofstream outputFragment;
  char FragmentName[MAXSTRINGSIZE];
  char PathBackup[MAXSTRINGSIZE];
  bool result = true;
  bool intermediateResult = true;
  atom *Walker = NULL;
  Vector BoxDimension;
  char *FragmentNumber = NULL;
  char *path = NULL;
  int FragmentCounter = 0;
  ofstream output;

  // store the fragments as config and as xyz
  for (MoleculeList::iterator ListRunner = ListOfMolecules.begin(); ListRunner != ListOfMolecules.end(); ListRunner++) {
    // save default path as it is changed for each fragment
    path = configuration->GetDefaultPath();
    if (path != NULL)
      strcpy(PathBackup, path);
    else {
      eLog() << Verbose(0) << "OutputConfigForListOfFragments: NULL default path obtained from config!" << endl;
      performCriticalExit();
    }

    // correct periodic
    (*ListRunner)->ScanForPeriodicCorrection();

    // output xyz file
    FragmentNumber = FixedDigitNumber(ListOfMolecules.size(), FragmentCounter++);
    sprintf(FragmentName, "%s/%s%s.conf.xyz", configuration->configpath, FRAGMENTPREFIX, FragmentNumber);
    outputFragment.open(FragmentName, ios::out);
    Log() << Verbose(2) << "Saving bond fragment No. " << FragmentNumber << "/" << FragmentCounter - 1 << " as XYZ ...";
    if ((intermediateResult = (*ListRunner)->OutputXYZ(&outputFragment)))
      Log() << Verbose(0) << " done." << endl;
    else
      Log() << Verbose(0) << " failed." << endl;
    result = result && intermediateResult;
    outputFragment.close();
    outputFragment.clear();

    // list atoms in fragment for debugging
    Log() << Verbose(2) << "Contained atoms: ";
    Walker = (*ListRunner)->start;
    while (Walker->next != (*ListRunner)->end) {
      Walker = Walker->next;
      Log() << Verbose(0) << Walker->Name << " ";
    }
    Log() << Verbose(0) << endl;

    // center on edge
    (*ListRunner)->CenterEdge(&BoxDimension);
    (*ListRunner)->SetBoxDimension(&BoxDimension); // update Box of atoms by boundary
    int j = -1;
    for (int k = 0; k < NDIM; k++) {
      j += k + 1;
      BoxDimension.x[k] = 2.5 * (configuration->GetIsAngstroem() ? 1. : 1. / AtomicLengthToAngstroem);
      (*ListRunner)->cell_size[j] += BoxDimension.x[k] * 2.;
    }
    (*ListRunner)->Translate(&BoxDimension);

    // also calculate necessary orbitals
    (*ListRunner)->CountElements(); // this is a bugfix, atoms should shoulds actually be added correctly to this fragment
    (*ListRunner)->CalculateOrbitals(*configuration);

    // change path in config
    //strcpy(PathBackup, configuration->configpath);
    sprintf(FragmentName, "%s/%s%s/", PathBackup, FRAGMENTPREFIX, FragmentNumber);
    configuration->SetDefaultPath(FragmentName);

    // and save as config
    sprintf(FragmentName, "%s/%s%s.conf", configuration->configpath, FRAGMENTPREFIX, FragmentNumber);
    Log() << Verbose(2) << "Saving bond fragment No. " << FragmentNumber << "/" << FragmentCounter - 1 << " as config ...";
    if ((intermediateResult = configuration->Save(FragmentName, (*ListRunner)->elemente, (*ListRunner))))
      Log() << Verbose(0) << " done." << endl;
    else
      Log() << Verbose(0) << " failed." << endl;
    result = result && intermediateResult;

    // restore old config
    configuration->SetDefaultPath(PathBackup);

    // and save as mpqc input file
    sprintf(FragmentName, "%s/%s%s.conf", configuration->configpath, FRAGMENTPREFIX, FragmentNumber);
    Log() << Verbose(2) << "Saving bond fragment No. " << FragmentNumber << "/" << FragmentCounter - 1 << " as mpqc input ...";
    if ((intermediateResult = configuration->SaveMPQC(FragmentName, (*ListRunner))))
      Log() << Verbose(2) << " done." << endl;
    else
      Log() << Verbose(0) << " failed." << endl;

    result = result && intermediateResult;
    //outputFragment.close();
    //outputFragment.clear();
    Free(&FragmentNumber);
  }
  Log() << Verbose(0) << " done." << endl;

  // printing final number
  Log() << Verbose(2) << "Final number of fragments: " << FragmentCounter << "." << endl;

  return result;
};

/** Counts the number of molecules with the molecule::ActiveFlag set.
 * \return number of molecules with ActiveFlag set to true.
 */
int MoleculeListClass::NumberOfActiveMolecules()
{
  int count = 0;
  for (MoleculeList::iterator ListRunner = ListOfMolecules.begin(); ListRunner != ListOfMolecules.end(); ListRunner++)
    count += ((*ListRunner)->ActiveFlag ? 1 : 0);
  return count;
};

/** Dissects given \a *mol into connected subgraphs and inserts them as new molecules but with old atoms into \a this.
 * \param *out output stream for debugging
 * \param *periode periodentafel
 * \param *configuration config with BondGraph
 */
void MoleculeListClass::DissectMoleculeIntoConnectedSubgraphs(const periodentafel * const periode, config * const configuration)
{
  molecule *mol = new molecule(periode);
  atom *Walker = NULL;
  atom *Advancer = NULL;
  bond *Binder = NULL;
  bond *Stepper = NULL;
  // 0. gather all atoms into single molecule
  for (MoleculeList::iterator MolRunner = ListOfMolecules.begin(); !ListOfMolecules.empty(); MolRunner = ListOfMolecules.begin()) {
    // shift all atoms to new molecule
    Advancer = (*MolRunner)->start->next;
    while (Advancer != (*MolRunner)->end) {
      Walker = Advancer;
      Advancer = Advancer->next;
      Log() << Verbose(3) << "Re-linking " << *Walker << "..." << endl;
      unlink(Walker);
      Walker->father = Walker;
      mol->AddAtom(Walker);    // counting starts at 1
    }
    // remove all bonds
    Stepper = (*MolRunner)->first->next;
    while (Stepper != (*MolRunner)->last) {
      Binder = Stepper;
      Stepper = Stepper->next;
      delete(Binder);
    }
    // remove the molecule
    delete(*MolRunner);
    ListOfMolecules.erase(MolRunner);
  }

  // 1. dissect the molecule into connected subgraphs
  configuration->BG->ConstructBondGraph(mol);

  // 2. scan for connected subgraphs
  MoleculeLeafClass *Subgraphs = NULL;      // list of subgraphs from DFS analysis
  class StackClass<bond *> *BackEdgeStack = NULL;
  Subgraphs = mol->DepthFirstSearchAnalysis(BackEdgeStack);
  delete(BackEdgeStack);

  // 3. dissect (the following construct is needed to have the atoms not in the order of the DFS, but in
  // the original one as parsed in)
  // TODO: Optimize this, when molecules just contain pointer list of global atoms!

  // 4a. create array of molecules to fill
  const int MolCount = Subgraphs->next->Count();
  char number[MAXSTRINGSIZE];
  molecule **molecules = Malloc<molecule *>(MolCount, "config::Load() - **molecules");
  for (int i=0;i<MolCount;i++) {
    molecules[i] = (molecule*) new molecule(mol->elemente);
    molecules[i]->ActiveFlag = true;
    strncpy(molecules[i]->name, mol->name, MAXSTRINGSIZE);
    if (MolCount > 1) {
      sprintf(number, "-%d", i+1);
      strncat(molecules[i]->name, number, MAXSTRINGSIZE - strlen(mol->name) - 1);
    }
    cout << "MolName is " << molecules[i]->name << endl;
    insert(molecules[i]);
  }

  // 4b. create and fill map of which atom is associated to which connected molecule (note, counting starts at 1)
  int FragmentCounter = 0;
  int *MolMap = Calloc<int>(mol->AtomCount, "config::Load() - *MolMap");
  MoleculeLeafClass *MolecularWalker = Subgraphs;
  Walker = NULL;
  while (MolecularWalker->next != NULL) {
    MolecularWalker = MolecularWalker->next;
    Walker = MolecularWalker->Leaf->start;
    while (Walker->next != MolecularWalker->Leaf->end) {
      Walker = Walker->next;
      MolMap[Walker->GetTrueFather()->nr] = FragmentCounter+1;
    }
    FragmentCounter++;
  }

  // 4c. relocate atoms to new molecules and remove from Leafs
  Walker = NULL;
  while (mol->start->next != mol->end) {
    Walker = mol->start->next;
    if ((Walker->nr <0) || (Walker->nr >= mol->AtomCount)) {
      eLog() << Verbose(0) << "Index of atom " << *Walker << " is invalid!" << endl;
      performCriticalExit();
    }
    FragmentCounter = MolMap[Walker->nr];
    if (FragmentCounter != 0) {
      Log() << Verbose(3) << "Re-linking " << *Walker << "..." << endl;
      unlink(Walker);
      molecules[FragmentCounter-1]->AddAtom(Walker);    // counting starts at 1
    } else {
      eLog() << Verbose(0) << "Atom " << *Walker << " not associated to molecule!" << endl;
      performCriticalExit();
    }
  }
  // 4d. we don't need to redo bonds, as they are connected subgraphs and still maintain their ListOfBonds, but we have to remove them from first..last list
  Binder = mol->first;
  while (mol->first->next != mol->last) {
    Binder = mol->first->next;
    Walker = Binder->leftatom;
    unlink(Binder);
    link(Binder,molecules[MolMap[Walker->nr]-1]->last);   // counting starts at 1
  }
  // 4e. free Leafs
  MolecularWalker = Subgraphs;
  while (MolecularWalker->next != NULL) {
    MolecularWalker = MolecularWalker->next;
    delete(MolecularWalker->previous);
  }
  delete(MolecularWalker);
  Free(&MolMap);
  Free(&molecules);
  Log() << Verbose(1) << "I scanned " << FragmentCounter << " molecules." << endl;
};

/** Count all atoms in each molecule.
 * \return number of atoms in the MoleculeListClass.
 * TODO: the inner loop should be done by some (double molecule::CountAtom()) function
 */
int MoleculeListClass::CountAllAtoms() const
{
  atom *Walker = NULL;
  int AtomNo = 0;
  for (MoleculeList::const_iterator MolWalker = ListOfMolecules.begin(); MolWalker != ListOfMolecules.end(); MolWalker++) {
    Walker = (*MolWalker)->start;
    while (Walker->next != (*MolWalker)->end) {
      Walker = Walker->next;
      AtomNo++;
    }
  }
  return AtomNo;
}


/******************************************* Class MoleculeLeafClass ************************************************/

/** Constructor for MoleculeLeafClass root leaf.
 * \param *Up Leaf on upper level
 * \param *PreviousLeaf NULL - We are the first leaf on this level, otherwise points to previous in list
 */
//MoleculeLeafClass::MoleculeLeafClass(MoleculeLeafClass *Up = NULL, MoleculeLeafClass *Previous = NULL)
MoleculeLeafClass::MoleculeLeafClass(MoleculeLeafClass *PreviousLeaf = NULL)
{
  //  if (Up != NULL)
  //    if (Up->DownLeaf == NULL) // are we the first down leaf for the upper leaf?
  //      Up->DownLeaf = this;
  //  UpLeaf = Up;
  //  DownLeaf = NULL;
  Leaf = NULL;
  previous = PreviousLeaf;
  if (previous != NULL) {
    MoleculeLeafClass *Walker = previous->next;
    previous->next = this;
    next = Walker;
  } else {
    next = NULL;
  }
};

/** Destructor for MoleculeLeafClass.
 */
MoleculeLeafClass::~MoleculeLeafClass()
{
  //  if (DownLeaf != NULL) {// drop leaves further down
  //    MoleculeLeafClass *Walker = DownLeaf;
  //    MoleculeLeafClass *Next;
  //    do {
  //      Next = Walker->NextLeaf;
  //      delete(Walker);
  //      Walker = Next;
  //    } while (Walker != NULL);
  //    // Last Walker sets DownLeaf automatically to NULL
  //  }
  // remove the leaf itself
  if (Leaf != NULL) {
    delete (Leaf);
    Leaf = NULL;
  }
  // remove this Leaf from level list
  if (previous != NULL)
    previous->next = next;
  //  } else { // we are first in list (connects to UpLeaf->DownLeaf)
  //    if ((NextLeaf != NULL) && (NextLeaf->UpLeaf == NULL))
  //      NextLeaf->UpLeaf = UpLeaf;  // either null as we are top level or the upleaf of the first node
  //    if (UpLeaf != NULL)
  //      UpLeaf->DownLeaf = NextLeaf;  // either null as we are only leaf or NextLeaf if we are just the first
  //  }
  //  UpLeaf = NULL;
  if (next != NULL) // are we last in list
    next->previous = previous;
  next = NULL;
  previous = NULL;
};

/** Adds \a molecule leaf to the tree.
 * \param *ptr ptr to molecule to be added
 * \param *Previous previous MoleculeLeafClass referencing level and which on the level
 * \return true - success, false - something went wrong
 */
bool MoleculeLeafClass::AddLeaf(molecule *ptr, MoleculeLeafClass *Previous)
{
  return false;
};

/** Fills the bond structure of this chain list subgraphs that are derived from a complete \a *reference molecule.
 * Calls this routine in each MoleculeLeafClass::next subgraph if it's not NULL.
 * \param *out output stream for debugging
 * \param *reference reference molecule with the bond structure to be copied
 * \param &FragmentCounter Counter needed to address \a **ListOfLocalAtoms
 * \param ***ListOfLocalAtoms Lookup table for each subgraph and index of each atom in \a *reference, may be NULL on start, then it is filled
 * \param FreeList true - ***ListOfLocalAtoms is free'd before return, false - it is not
 * \return true - success, false - faoilure
 */
bool MoleculeLeafClass::FillBondStructureFromReference(const molecule * const reference, int &FragmentCounter, atom ***&ListOfLocalAtoms, bool FreeList)
{
  atom *Walker = NULL;
  atom *OtherWalker = NULL;
  atom *Father = NULL;
  bool status = true;
  int AtomNo;

  Log() << Verbose(1) << "Begin of FillBondStructureFromReference." << endl;
  // fill ListOfLocalAtoms if NULL was given
  if (!FillListOfLocalAtoms(ListOfLocalAtoms, FragmentCounter, reference->AtomCount, FreeList)) {
    Log() << Verbose(1) << "Filling of ListOfLocalAtoms failed." << endl;
    return false;
  }

  if (status) {
    Log() << Verbose(1) << "Creating adjacency list for subgraph " << Leaf << "." << endl;
    // remove every bond from the list
    bond *Binder = NULL;
    while (Leaf->last->previous != Leaf->first) {
      Binder = Leaf->last->previous;
      Binder->leftatom->UnregisterBond(Binder);
      Binder->rightatom->UnregisterBond(Binder);
      removewithoutcheck(Binder);
    }

    Walker = Leaf->start;
    while (Walker->next != Leaf->end) {
      Walker = Walker->next;
      Father = Walker->GetTrueFather();
      AtomNo = Father->nr; // global id of the current walker
      for (BondList::const_iterator Runner = Father->ListOfBonds.begin(); Runner != Father->ListOfBonds.end(); (++Runner)) {
        OtherWalker = ListOfLocalAtoms[FragmentCounter][(*Runner)->GetOtherAtom(Walker->GetTrueFather())->nr]; // local copy of current bond partner of walker
        if (OtherWalker != NULL) {
          if (OtherWalker->nr > Walker->nr)
            Leaf->AddBond(Walker, OtherWalker, (*Runner)->BondDegree);
        } else {
          Log() << Verbose(1) << "OtherWalker = ListOfLocalAtoms[" << FragmentCounter << "][" << (*Runner)->GetOtherAtom(Walker->GetTrueFather())->nr << "] is NULL!" << endl;
          status = false;
        }
      }
    }
  }

  if ((FreeList) && (ListOfLocalAtoms != NULL)) {
    // free the index lookup list
    Free(&ListOfLocalAtoms[FragmentCounter]);
    if (FragmentCounter == 0) // first fragments frees the initial pointer to list
      Free(&ListOfLocalAtoms);
  }
  Log() << Verbose(1) << "End of FillBondStructureFromReference." << endl;
  return status;
};

/** Fills the root stack for sites to be used as root in fragmentation depending on order or adaptivity criteria
 * Again, as in \sa FillBondStructureFromReference steps recursively through each Leaf in this chain list of molecule's.
 * \param *out output stream for debugging
 * \param *&RootStack stack to be filled
 * \param *AtomMask defines true/false per global Atom::nr to mask in/out each nuclear site
 * \param &FragmentCounter counts through the fragments in this MoleculeLeafClass
 * \return true - stack is non-empty, fragmentation necessary, false - stack is empty, no more sites to update
 */
bool MoleculeLeafClass::FillRootStackForSubgraphs(KeyStack *&RootStack, bool *AtomMask, int &FragmentCounter)
{
  atom *Walker = NULL, *Father = NULL;

  if (RootStack != NULL) {
    // find first root candidates
    if (&(RootStack[FragmentCounter]) != NULL) {
      RootStack[FragmentCounter].clear();
      Walker = Leaf->start;
      while (Walker->next != Leaf->end) { // go through all (non-hydrogen) atoms
        Walker = Walker->next;
        Father = Walker->GetTrueFather();
        if (AtomMask[Father->nr]) // apply mask
#ifdef ADDHYDROGEN
          if (Walker->type->Z != 1) // skip hydrogen
#endif
          RootStack[FragmentCounter].push_front(Walker->nr);
      }
      if (next != NULL)
        next->FillRootStackForSubgraphs(RootStack, AtomMask, ++FragmentCounter);
    } else {
      Log() << Verbose(1) << "Rootstack[" << FragmentCounter << "] is NULL." << endl;
      return false;
    }
    FragmentCounter--;
    return true;
  } else {
    Log() << Verbose(1) << "Rootstack is NULL." << endl;
    return false;
  }
};

/** Fills a lookup list of father's Atom::nr -> atom for each subgraph.
 * \param *out output stream from debugging
 * \param ***ListOfLocalAtoms Lookup table for each subgraph and index of each atom in global molecule, may be NULL on start, then it is filled
 * \param FragmentCounter counts the fragments as we move along the list
 * \param GlobalAtomCount number of atoms in the complete molecule
 * \param &FreeList true - ***ListOfLocalAtoms is free'd before return, false - it is not
 * \return true - success, false - failure
 */
bool MoleculeLeafClass::FillListOfLocalAtoms(atom ***&ListOfLocalAtoms, const int FragmentCounter, const int GlobalAtomCount, bool &FreeList)
{
  bool status = true;

  if (ListOfLocalAtoms == NULL) { // allocated initial pointer
    // allocate and set each field to NULL
    const int Counter = Count();
    ListOfLocalAtoms = Calloc<atom**>(Counter, "MoleculeLeafClass::FillListOfLocalAtoms - ***ListOfLocalAtoms");
    if (ListOfLocalAtoms == NULL) {
      FreeList = FreeList && false;
      status = false;
    }
  }

  if ((ListOfLocalAtoms != NULL) && (ListOfLocalAtoms[FragmentCounter] == NULL)) { // allocate and fill list of this fragment/subgraph
    status = status && CreateFatherLookupTable(Leaf->start, Leaf->end, ListOfLocalAtoms[FragmentCounter], GlobalAtomCount);
    FreeList = FreeList && true;
  }

  return status;
};

/** The indices per keyset are compared to the respective father's Atom::nr in each subgraph and thus put into \a **&FragmentList.
 * \param *out output stream fro debugging
 * \param *reference reference molecule with the bond structure to be copied
 * \param *KeySetList list with all keysets
 * \param ***ListOfLocalAtoms Lookup table for each subgraph and index of each atom in global molecule, may be NULL on start, then it is filled
 * \param **&FragmentList list to be allocated and returned
 * \param &FragmentCounter counts the fragments as we move along the list
 * \param FreeList true - ***ListOfLocalAtoms is free'd before return, false - it is not
 * \retuen true - success, false - failure
 */
bool MoleculeLeafClass::AssignKeySetsToFragment(molecule *reference, Graph *KeySetList, atom ***&ListOfLocalAtoms, Graph **&FragmentList, int &FragmentCounter, bool FreeList)
{
  bool status = true;
  int KeySetCounter = 0;

  Log() << Verbose(1) << "Begin of AssignKeySetsToFragment." << endl;
  // fill ListOfLocalAtoms if NULL was given
  if (!FillListOfLocalAtoms(ListOfLocalAtoms, FragmentCounter, reference->AtomCount, FreeList)) {
    Log() << Verbose(1) << "Filling of ListOfLocalAtoms failed." << endl;
    return false;
  }

  // allocate fragment list
  if (FragmentList == NULL) {
    KeySetCounter = Count();
    FragmentList = Calloc<Graph*>(KeySetCounter, "MoleculeLeafClass::AssignKeySetsToFragment - **FragmentList");
    KeySetCounter = 0;
  }

  if ((KeySetList != NULL) && (KeySetList->size() != 0)) { // if there are some scanned keysets at all
    // assign scanned keysets
    if (FragmentList[FragmentCounter] == NULL)
      FragmentList[FragmentCounter] = new Graph;
    KeySet *TempSet = new KeySet;
    for (Graph::iterator runner = KeySetList->begin(); runner != KeySetList->end(); runner++) { // key sets contain global numbers!
      if (ListOfLocalAtoms[FragmentCounter][reference->FindAtom(*((*runner).first.begin()))->nr] != NULL) {// as we may assume that that bond structure is unchanged, we only test the first key in each set
        // translate keyset to local numbers
        for (KeySet::iterator sprinter = (*runner).first.begin(); sprinter != (*runner).first.end(); sprinter++)
          TempSet->insert(ListOfLocalAtoms[FragmentCounter][reference->FindAtom(*sprinter)->nr]->nr);
        // insert into FragmentList
        FragmentList[FragmentCounter]->insert(GraphPair(*TempSet, pair<int, double> (KeySetCounter++, (*runner).second.second)));
      }
      TempSet->clear();
    }
    delete (TempSet);
    if (KeySetCounter == 0) {// if there are no keysets, delete the list
      Log() << Verbose(1) << "KeySetCounter is zero, deleting FragmentList." << endl;
      delete (FragmentList[FragmentCounter]);
    } else
      Log() << Verbose(1) << KeySetCounter << " keysets were assigned to subgraph " << FragmentCounter << "." << endl;
    FragmentCounter++;
    if (next != NULL)
      next->AssignKeySetsToFragment(reference, KeySetList, ListOfLocalAtoms, FragmentList, FragmentCounter, FreeList);
    FragmentCounter--;
  } else
    Log() << Verbose(1) << "KeySetList is NULL or empty." << endl;

  if ((FreeList) && (ListOfLocalAtoms != NULL)) {
    // free the index lookup list
    Free(&ListOfLocalAtoms[FragmentCounter]);
    if (FragmentCounter == 0) // first fragments frees the initial pointer to list
      Free(&ListOfLocalAtoms);
  }
  Log() << Verbose(1) << "End of AssignKeySetsToFragment." << endl;
  return status;
};

/** Translate list into global numbers (i.e. ones that are valid in "this" molecule, not in MolecularWalker->Leaf)
 * \param *out output stream for debugging
 * \param **FragmentList Graph with local numbers per fragment
 * \param &FragmentCounter counts the fragments as we move along the list
 * \param &TotalNumberOfKeySets global key set counter
 * \param &TotalGraph Graph to be filled with global numbers
 */
void MoleculeLeafClass::TranslateIndicesToGlobalIDs(Graph **FragmentList, int &FragmentCounter, int &TotalNumberOfKeySets, Graph &TotalGraph)
{
  Log() << Verbose(1) << "Begin of TranslateIndicesToGlobalIDs." << endl;
  KeySet *TempSet = new KeySet;
  if (FragmentList[FragmentCounter] != NULL) {
    for (Graph::iterator runner = FragmentList[FragmentCounter]->begin(); runner != FragmentList[FragmentCounter]->end(); runner++) {
      for (KeySet::iterator sprinter = (*runner).first.begin(); sprinter != (*runner).first.end(); sprinter++)
        TempSet->insert((Leaf->FindAtom(*sprinter))->GetTrueFather()->nr);
      TotalGraph.insert(GraphPair(*TempSet, pair<int, double> (TotalNumberOfKeySets++, (*runner).second.second)));
      TempSet->clear();
    }
    delete (TempSet);
  } else {
    Log() << Verbose(1) << "FragmentList is NULL." << endl;
  }
  if (next != NULL)
    next->TranslateIndicesToGlobalIDs(FragmentList, ++FragmentCounter, TotalNumberOfKeySets, TotalGraph);
  FragmentCounter--;
  Log() << Verbose(1) << "End of TranslateIndicesToGlobalIDs." << endl;
};

/** Simply counts the number of items in the list, from given MoleculeLeafClass.
 * \return number of items
 */
int MoleculeLeafClass::Count() const
{
  if (next != NULL)
    return next->Count() + 1;
  else
    return 1;
};

