/*
 * unittest.cpp
 *
 *  Created on: Aug 17, 2009
 *      Author: heber
 */


using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "vectorunittest.hpp"
#include "vector.hpp"
#include "defs.hpp"

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( VectorTest );


void VectorTest::setUp()
{
  zero.Init(0.,0.,0.);
  unit.Init(1.,0.,0.);
  otherunit.Init(0.,1.,0.);
  notunit.Init(0.,1.,1.);
  two.Init(2.,1.,0.);
};


void VectorTest::tearDown()
{
};

/** UnitTest for Constructors and Vector::IsZero() and Vector::IsOne().
 */
void VectorTest::UnityTest()
{
  // unity and zero tests
  CPPUNIT_ASSERT_EQUAL( true, zero.IsZero() );
  CPPUNIT_ASSERT_EQUAL( false, zero.IsOne() );
  CPPUNIT_ASSERT_EQUAL( false, unit.IsZero() );
  CPPUNIT_ASSERT_EQUAL( true, unit.IsOne() );
  CPPUNIT_ASSERT_EQUAL( false, notunit.IsOne() );
  CPPUNIT_ASSERT_EQUAL( true, otherunit.IsOne() );
  CPPUNIT_ASSERT_EQUAL( false, otherunit.IsZero() );
};

/** UnitTest for Vector::CopyVector(), Vector::AddVector, Vector::SubtractVector() and Vector::Scale()/
 */
void VectorTest::SimpleAlgebraTest()
{
  double factor;
  // copy vector
  fixture.Init(2.,3.,4.);
  CPPUNIT_ASSERT_EQUAL( Vector(2.,3.,4.), fixture );
  // summation and scaling
  fixture.CopyVector(&zero);
  fixture.AddVector(&unit);
  CPPUNIT_ASSERT_EQUAL( true, fixture.IsOne() );
  fixture.CopyVector(&zero);
  fixture.SubtractVector(&unit);
  CPPUNIT_ASSERT_EQUAL( true, fixture.IsOne() );
  CPPUNIT_ASSERT_EQUAL( false, fixture.IsZero() );
  fixture.CopyVector(&zero);
  fixture.AddVector(&zero);
  CPPUNIT_ASSERT_EQUAL( true, fixture.IsZero() );
  fixture.CopyVector(&notunit);
  fixture.SubtractVector(&otherunit);
  CPPUNIT_ASSERT_EQUAL( true, fixture.IsOne() );
  fixture.CopyVector(&unit);
  fixture.AddVector(&otherunit);
  CPPUNIT_ASSERT_EQUAL( false, fixture.IsOne() );
  fixture.CopyVector(&notunit);
  fixture.SubtractVector(&unit);
  fixture.SubtractVector(&otherunit);
  CPPUNIT_ASSERT_EQUAL( false, fixture.IsZero() );
  fixture.CopyVector(&unit);
  fixture.Scale(0.98);
  CPPUNIT_ASSERT_EQUAL( false, fixture.IsOne() );
  fixture.CopyVector(&unit);
  fixture.Scale(1.);
  CPPUNIT_ASSERT_EQUAL( true, fixture.IsOne() );
  fixture.CopyVector(&unit);
  factor = 0.98;
  fixture.Scale(factor);
  CPPUNIT_ASSERT_EQUAL( false, fixture.IsOne() );
  fixture.CopyVector(&unit);
  factor = 1.;
  fixture.Scale(factor);
  CPPUNIT_ASSERT_EQUAL( true, fixture.IsOne() );
};


/** UnitTest for operator versions of Vector::CopyVector(), Vector::AddVector, Vector::SubtractVector() and Vector::Scale().
 */
void VectorTest::OperatorAlgebraTest()
{
  // summation and scaling
  CPPUNIT_ASSERT_EQUAL( true, (zero+unit).IsOne() );
  CPPUNIT_ASSERT_EQUAL( true, (zero+unit).IsOne() );
  CPPUNIT_ASSERT_EQUAL( true, (zero-unit).IsOne() );
  CPPUNIT_ASSERT_EQUAL( false, (zero-unit).IsZero() );
  CPPUNIT_ASSERT_EQUAL( true, (zero+zero).IsZero() );
  CPPUNIT_ASSERT_EQUAL( true, (notunit-otherunit).IsOne() );
  CPPUNIT_ASSERT_EQUAL( false, (unit+otherunit).IsOne() );
  CPPUNIT_ASSERT_EQUAL( false, (notunit-unit-otherunit).IsZero() );
  CPPUNIT_ASSERT_EQUAL( false, (unit*0.98).IsOne() );
  CPPUNIT_ASSERT_EQUAL( true, (unit*1.).IsOne() );

  CPPUNIT_ASSERT_EQUAL( unit, (zero+unit) );
  CPPUNIT_ASSERT_EQUAL( Vector(0.,0.,1.), (notunit-otherunit) );
  CPPUNIT_ASSERT_EQUAL( Vector(-1, 0., 1.), (notunit-unit-otherunit) );
};

/** UnitTest for scalar products.
 */
void VectorTest::EuclidianScalarProductTest()
{
  CPPUNIT_ASSERT_EQUAL( 0., zero.ScalarProduct(&zero) );
  CPPUNIT_ASSERT_EQUAL( 0., zero.ScalarProduct(&unit) );
  CPPUNIT_ASSERT_EQUAL( 0., zero.ScalarProduct(&otherunit) );
  CPPUNIT_ASSERT_EQUAL( 0., zero.ScalarProduct(&notunit) );
  CPPUNIT_ASSERT_EQUAL( 1., unit.ScalarProduct(&unit) );
  CPPUNIT_ASSERT_EQUAL( 0., otherunit.ScalarProduct(&unit) );
  CPPUNIT_ASSERT_EQUAL( 0., otherunit.ScalarProduct(&unit) );
  CPPUNIT_ASSERT_EQUAL( 1., otherunit.ScalarProduct(&notunit) );
  CPPUNIT_ASSERT_EQUAL( 2., two.ScalarProduct(&unit) );
  CPPUNIT_ASSERT_EQUAL( 1., two.ScalarProduct(&otherunit) );
  CPPUNIT_ASSERT_EQUAL( 1., two.ScalarProduct(&notunit) );
}

/** UnitTest for norms.
 */
void VectorTest::EuclidianNormTest()
{
  CPPUNIT_ASSERT_EQUAL( 0., zero.Norm() );
  CPPUNIT_ASSERT_EQUAL( 0., zero.NormSquared() );
  CPPUNIT_ASSERT_EQUAL( 1., unit.Norm() );
  CPPUNIT_ASSERT_EQUAL( 1., unit.NormSquared() );
  CPPUNIT_ASSERT_EQUAL( 1., otherunit.Norm() );
  CPPUNIT_ASSERT_EQUAL( 1., otherunit.NormSquared() );
  CPPUNIT_ASSERT_EQUAL( 2., notunit.NormSquared() );
  CPPUNIT_ASSERT_EQUAL( sqrt(2.), notunit.Norm() );
}

/** UnitTest for distances.
 */
void VectorTest::EuclidianDistancesTest()
{
  CPPUNIT_ASSERT_EQUAL( 1., zero.Distance(&unit) );
  CPPUNIT_ASSERT_EQUAL( sqrt(2.), otherunit.Distance(&unit) );
  CPPUNIT_ASSERT_EQUAL( sqrt(2.), zero.Distance(&notunit) );
  CPPUNIT_ASSERT_EQUAL( 1., otherunit.Distance(&notunit) );
  CPPUNIT_ASSERT_EQUAL( sqrt(5.), two.Distance(&notunit) );
}

/** UnitTest for angles.
 */
void VectorTest::EuclidianAnglesTest()
{
  CPPUNIT_ASSERT_EQUAL( M_PI, zero.Angle(&unit) );
  CPPUNIT_ASSERT_EQUAL( 0., unit.Angle(&unit) );
  CPPUNIT_ASSERT_EQUAL( true, fabs(M_PI/2. - otherunit.Angle(&unit)) < MYEPSILON );
  CPPUNIT_ASSERT_EQUAL( true, fabs(M_PI/2. - unit.Angle(&notunit)) < MYEPSILON );
  CPPUNIT_ASSERT_EQUAL( true, fabs(M_PI/4. - otherunit.Angle(&notunit)) < MYEPSILON );
};

/** UnitTest for projections.
 */
void VectorTest::ProjectionTest()
{
  CPPUNIT_ASSERT_EQUAL( zero, zero.Projection(&unit) );
  CPPUNIT_ASSERT_EQUAL( zero, otherunit.Projection(&unit) );
  CPPUNIT_ASSERT_EQUAL( Vector(0.4,0.2,0.),  otherunit.Projection(&two) );
  CPPUNIT_ASSERT_EQUAL( Vector(0.,1.,0.),  two.Projection(&otherunit) );
};

/** UnitTest for line intersections.
 */
void VectorTest::LineIntersectionTest()
{
  // plane at (0,0,0) normal to (1,0,0) cuts line from (0,0,0) to (2,1,0) at ???
  CPPUNIT_ASSERT_EQUAL( true, fixture.GetIntersectionWithPlane((ofstream *)&cout, &unit, &zero, &zero, &two) );
  CPPUNIT_ASSERT_EQUAL( zero, fixture );

  // plane at (2,1,0) normal to (0,1,0) cuts line from (1,0,0) to (0,1,1) at ???
  CPPUNIT_ASSERT_EQUAL( true, fixture.GetIntersectionWithPlane((ofstream *)&cout, &otherunit, &two, &unit, &notunit) );
  CPPUNIT_ASSERT_EQUAL( Vector(0., 1., 1.), fixture );

  // four vectors equal to zero
  CPPUNIT_ASSERT_EQUAL( false, fixture.GetIntersectionOfTwoLinesOnPlane((ofstream *)&cout, &zero, &zero, &zero, &zero, NULL) );
  CPPUNIT_ASSERT_EQUAL( zero, fixture );

  // four vectors equal to unit
  CPPUNIT_ASSERT_EQUAL( false, fixture.GetIntersectionOfTwoLinesOnPlane((ofstream *)&cout, &unit, &unit, &unit, &unit, NULL) );
  CPPUNIT_ASSERT_EQUAL( zero, fixture );

  // two equal lines
  CPPUNIT_ASSERT_EQUAL( true, fixture.GetIntersectionOfTwoLinesOnPlane((ofstream *)&cout, &unit, &two, &unit, &two, NULL) );
  CPPUNIT_ASSERT_EQUAL( unit, fixture );

  // line from (1,0,0) to (2,1,0) cuts line from (1,0,0) to (0,1,0) at ???
  CPPUNIT_ASSERT_EQUAL( true, fixture.GetIntersectionOfTwoLinesOnPlane((ofstream *)&cout, &unit, &two, &unit, &otherunit, NULL) );
  CPPUNIT_ASSERT_EQUAL( unit, fixture );

  // line from (1,0,0) to (0,0,0) cuts line from (0,0,0) to (2,1,0) at ???
  CPPUNIT_ASSERT_EQUAL( true, fixture.GetIntersectionOfTwoLinesOnPlane((ofstream *)&cout, &unit, &zero, &zero, &two, NULL) );
  CPPUNIT_ASSERT_EQUAL( zero, fixture );

  // line from (1,0,0) to (2,1,0) cuts line from (0,0,0) to (0,1,0) at ???
  CPPUNIT_ASSERT_EQUAL( true, fixture.GetIntersectionOfTwoLinesOnPlane((ofstream *)&cout, &unit, &two, &zero, &otherunit, NULL) );
  CPPUNIT_ASSERT_EQUAL( Vector(0., -1., 0.), fixture );
};

/** UnitTest for vector rotations.
 */
void VectorTest::VectorRotationTest()
{
  fixture.Init(-1.,0.,0.);

  // zero vector does not change
  fixture.CopyVector(&zero);
  fixture.RotateVector(&unit, 1.);
  CPPUNIT_ASSERT_EQUAL( zero, fixture );

  fixture.RotateVector(&two, 1.);
  CPPUNIT_ASSERT_EQUAL( zero,  fixture);

  // vector on axis does not change
  fixture.CopyVector(&unit);
  fixture.RotateVector(&unit, 1.);
  CPPUNIT_ASSERT_EQUAL( unit, fixture );

  fixture.RotateVector(&unit, 1.);
  CPPUNIT_ASSERT_EQUAL( unit, fixture );

  // rotations
  fixture.CopyVector(&otherunit);
  fixture.RotateVector(&unit, M_PI);
  CPPUNIT_ASSERT_EQUAL( Vector(0.,-1.,0.), fixture );

  fixture.CopyVector(&otherunit);
  fixture.RotateVector(&unit, 2. * M_PI);
  CPPUNIT_ASSERT_EQUAL( otherunit, fixture );

  fixture.CopyVector(&otherunit);
  fixture.RotateVector(&unit, 0);
  CPPUNIT_ASSERT_EQUAL( otherunit, fixture );

  fixture.Init(0.,0.,1.);
  fixture.RotateVector(&notunit, M_PI);
  CPPUNIT_ASSERT_EQUAL( otherunit, fixture );
}

/**
 * UnitTest for Vector::IsInParallelepiped().
 */
void VectorTest::IsInParallelepipedTest()
{
  double parallelepiped[NDIM*NDIM];
  parallelepiped[0] = 1;
  parallelepiped[1] = 0;
  parallelepiped[2] = 0;
  parallelepiped[3] = 0;
  parallelepiped[4] = 1;
  parallelepiped[5] = 0;
  parallelepiped[6] = 0;
  parallelepiped[7] = 0;
  parallelepiped[8] = 1;

  fixture.CopyVector(zero);
  CPPUNIT_ASSERT_EQUAL( false, fixture.IsInParallelepiped(Vector(2.,2.,2.), parallelepiped) );
  fixture.Init(2.5,2.5,2.5);
  CPPUNIT_ASSERT_EQUAL( true, fixture.IsInParallelepiped(Vector(2.,2.,2.), parallelepiped) );
  fixture.Init(1.,1.,1.);
  CPPUNIT_ASSERT_EQUAL( false, fixture.IsInParallelepiped(Vector(2.,2.,2.), parallelepiped) );
  fixture.Init(3.5,3.5,3.5);
  CPPUNIT_ASSERT_EQUAL( false, fixture.IsInParallelepiped(Vector(2.,2.,2.), parallelepiped) );
  fixture.Init(2.,2.,2.);
  CPPUNIT_ASSERT_EQUAL( true, fixture.IsInParallelepiped(Vector(2.,2.,2.), parallelepiped) );
  fixture.Init(2.,3.,2.);
  CPPUNIT_ASSERT_EQUAL( true, fixture.IsInParallelepiped(Vector(2.,2.,2.), parallelepiped) );
  fixture.Init(-2.,2.,-1.);
  CPPUNIT_ASSERT_EQUAL( false, fixture.IsInParallelepiped(Vector(2.,2.,2.), parallelepiped) );
}


/********************************************** Main routine **************************************/

int main(int argc, char **argv)
{
  // Get the top level suite from the registry
  CppUnit::Test *suite = CppUnit::TestFactoryRegistry::getRegistry().makeTest();

  // Adds the test to the list of test to run
  CppUnit::TextUi::TestRunner runner;
  runner.addTest( suite );

  // Change the default outputter to a compiler error format outputter
  runner.setOutputter( new CppUnit::CompilerOutputter( &runner.result(),
                                                       std::cerr ) );
  // Run the tests.
  bool wasSucessful = runner.run();

  // Return error code 1 if the one of test failed.
  return wasSucessful ? 0 : 1;
};
