/*
 * FragmentationResults.hpp
 *
 *  Created on: Aug 31, 2012
 *      Author: heber
 */

#ifndef FRAGMENTATIONRESULTS_HPP_
#define FRAGMENTATIONRESULTS_HPP_


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <map>
#include <vector>

#include "JobMarket/types.hpp"

#include "Fragmentation/KeySetsContainer.hpp"
#include "Fragmentation/Summation/IndexSet.hpp"
#include "Fragmentation/Summation/IndexSetContainer.hpp"
#include "Fragmentation/Summation/SubsetMap.hpp"

#include "Jobs/MPQCData.hpp"
#include "Jobs/MPQCDataMap.hpp"
#include "Jobs/VMGData.hpp"
#include "Jobs/VMGDataMap.hpp"
#include "Jobs/Grid/SamplingGrid.hpp"

/** FragmentationResults contains the summed up results per level resulting
 * from the fragmentation of the molecular system and clever combination of
 * fragmentary energies, forces, timings, and so on.
 *
 * This structure is mostly a storage wherein the summed up results are
 * contained for subsequent pretty printing and so on.
 *
 */
struct FragmentationResults
{
  /** Constructor for class FragmentationResults, based on KeySets.
   *
   * @param fragmentData results from short-range fragment calculations
   * @param longrangeData results from long-range fragment calculations
   * @param _KeySet KeySets of all (non-hydrogen) atoms
   * @param _ForceKeySet KeySets of all atoms except those added by saturation
   */
  FragmentationResults(
      const std::map<JobId_t,MPQCData> &fragmentData,
      std::map<JobId_t,VMGData> &longrangeData,
      const KeySetsContainer& _KeySet,
      const KeySetsContainer& _ForceKeySet);

  /** Performs the summation and fills all result vectors.
   *
   * @param fragmentData results from short-range fragment calculations
   * @param longrangeData results from long-range fragment calculations
   * @param fullsolutionData long-range solution
   * @param full_sample sampled density
   */
  void operator()(
      const std::map<JobId_t,MPQCData> &fragmentData,
      std::map<JobId_t,VMGData> &longrangeData,
      const std::vector<VMGData> &fullsolutionData,
      const std::vector<SamplingGrid> &full_sample);

  size_t getMaxLevel() const {
    return MaxLevel;
  }

  IndexSetContainer::Container_t getContainer() const {
    return container->getContainer();
  }

  const KeySetsContainer& getKeySet() const {
    return KeySet;
  }

  const KeySetsContainer& getForceKeySet() const {
    return ForceKeySet;
  }

private:
  std::map< JobId_t, size_t > MPQCMatrixNrLookup;
  std::map< JobId_t, size_t > VMGMatrixNrLookup;
  KeySetsContainer KeySet;
  KeySetsContainer ForceKeySet;
  IndexSetContainer::ptr container;
  SubsetMap::ptr subsetmap;

public:
  //!> results per level of summed up energy
  std::vector<MPQCDataEnergyMap_t> Result_Energy_fused;
  //!> results per level of summed up sampled grid charge
  std::vector<MPQCDataGridMap_t> Result_Grid_fused;
  //!> results per level of summed up times
  std::vector<MPQCDataTimeMap_t> Result_Time_fused;
  //!> results per level of summed up fragment positions and charges
  std::vector<MPQCDataFragmentMap_t> Result_Fragment_fused;
  //!> results per level of summed up forces
  std::vector<MPQCDataForceMap_t> Result_Force_fused;
  //!> results per level of summed up long range potential grids and energy
  std::vector<VMGDataMap_t> Result_LongRange_fused;
  //!> results per level of summed up long range true energy
  std::vector<VMGDataLongRangeMap_t> Result_LongRangeIntegrated_fused;

  //!> results per IndexSet of summed up energy
  std::map<IndexSet::ptr, std::pair<MPQCDataEnergyMap_t,MPQCDataEnergyMap_t> > Result_perIndexSet_Energy;
  //!> results per IndexSet of summed up sampled grid charge
  std::map<IndexSet::ptr, std::pair<MPQCDataGridMap_t,MPQCDataGridMap_t> > Result_perIndexSet_Grid;
  //!> results per IndexSet of summed up times
  std::map<IndexSet::ptr, std::pair<MPQCDataTimeMap_t,MPQCDataTimeMap_t> > Result_perIndexSet_Time;
  //!> results per IndexSet of summed up fragment positions and charges
  std::map<IndexSet::ptr, std::pair<MPQCDataFragmentMap_t,MPQCDataFragmentMap_t> > Result_perIndexSet_Fragment;
  //!> results per IndexSet of summed up forces
  std::map<IndexSet::ptr, std::pair<MPQCDataForceMap_t, MPQCDataForceMap_t> > Result_perIndexSet_Force;
  //!> results per IndexSet of summed up long range potential grids and energy
  std::map<IndexSet::ptr, std::pair<VMGDataMap_t, VMGDataMap_t> > Result_perIndexSet_LongRange;
  // we don't need the map pendant for Result_LongRangeIntegrated_fused, as this
  // quantity makes sense only level-wise

private:
  //!> maximum level of summation
  size_t MaxLevel;
};


#endif /* FRAGMENTATIONRESULTS_HPP_ */
