/*
 * FragmentJobQueue.hpp
 *
 *  Created on: Mar 4, 2013
 *      Author: heber
 */

#ifndef FRAGMENTJOBQUEUE_HPP_
#define FRAGMENTJOBQUEUE_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <boost/filesystem/path.hpp>
#include <string>
#include <vector>

#include "CodePatterns/Singleton.hpp"

#include "Fragmentation/KeySetsContainer.hpp"
#include "Fragmentation/Automation/parseKeySetFile.hpp"
#include "Jobs/MPQCJob.hpp"

/** FragmentJobQueue is a static instance to contain all present fragment
 * as FragmentJob's and to allow their transfer from FragmentationAction
 * to FragmentationAutomationAction.
 *
 * \note All contained jobs do not have yet a valid id!
 */
class FragmentJobQueue : public Singleton<FragmentJobQueue>
{
  friend class Singleton<FragmentJobQueue>;
private:
  /** Private default constructor for class FragmentJobQueue.
   */
  FragmentJobQueue()
  {}

  /** Private destructor for class FragmentJobQueue.
   */
  ~FragmentJobQueue()
  {}

public:

  /** Pushes a vector of jobs into the queue.
   *
   * \param _jobs new jobs to push
   * \param KeySet KeySet of all (non-hydrogen) atoms
   * \param FullKeySet KeySet of all atoms except saturation hydrogens
   */
  void addJobs(
      std::vector<FragmentJob::ptr> &_jobs,
      const KeySetsContainer &_KeySets,
      const KeySetsContainer &_FullKeySets);

  /** Returns ref to jobs.
   *
   * \param vector with all jobs
   */
  const std::vector<FragmentJob::ptr>& getJobs() const { return jobs; }

  /** Getter for the number of jobs contained.
   *
   * \return number of jobs in queue
   */
  size_t size() const { return jobs.size(); }

  /** Parses multiple files.
   *
   * \param jobfiles vector of filenames to parse
   * \param level level for density sampling grid
   */
  bool addJobsFromFiles(
      const std::vector< boost::filesystem::path > &jobfiles,
      const unsigned int level);

  /** Adds keysets after fragments have been added by file.
   *
   */
  bool addKeySetsFromFiles(
      const boost::filesystem::path &path,
      const size_t FragmentCounter,
      const enum KeySetFileType keysettype
      );

  /** Adds keysets after fragments have been added by file.
   *
   */
  bool addFullKeySetsFromFiles(
      const boost::filesystem::path &path,
      const size_t FragmentCounter,
      const enum KeySetFileType keysettype
      );

  /** Getter for the container of all KeySets to the jobs.
   *
   * \return const ref to container
   */
  const KeySetsContainer& getKeySets() const { return KeySets; }

  /** Getter for the container of all FullKeySets to the jobs.
   *
   * \return const ref to container
   */
  const KeySetsContainer& getFullKeySets() const { return FullKeySets; }

  /* Empties contained jobs and all KeySets.
   *
   */
  void clear();

private:
  /** Creates a MPQCJob with argument \a filename.
   *
   * \note job is created with invalid id.
   *
   * @param filename filename being argument to job
   * @param level level for density sampling grid
   */
  void parsejob(const std::string &filename, const unsigned int level);

private:
  //!> container for all present jobs
  std::vector<FragmentJob::ptr> jobs;
  //!> container for all KeySet's without hydrogens to the jobs
  KeySetsContainer KeySets;
  //!> container for all KeySet's with all except saturation hydrogen to the jobs
  KeySetsContainer FullKeySets;
};

inline
void FragmentJobQueue::addJobs(
    std::vector<FragmentJob::ptr> &_jobs,
    const KeySetsContainer &_KeySets,
    const KeySetsContainer &_FullKeySets)
{
  jobs.insert(jobs.end(), _jobs.begin(), _jobs.end());
  KeySets.insert(_KeySets);
  FullKeySets.insert(_FullKeySets);
}

#endif /* FRAGMENTJOBQUEUE_HPP_ */
