/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * IsVoidNode_FillPredicate.cpp
 *
 *  Created on: Jan 16, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include <boost/bind.hpp>

#include "IsVoidNode_FillPredicate.hpp"

#include "CodePatterns/Log.hpp"

#include "Atom/TesselPoint.hpp"
#include "LinkedCell/LinkedCell_View.hpp"
#include "Shapes/ShapeOps.hpp"
#include "Tesselation/tesselation.hpp"
#include "World.hpp"

#include "AnyFillPredicate.hpp"
#include "FillPredicate.hpp"

/** Checks whether a given node \a n in the domain is empty with respect to the
 * given \a shape.
 *
 * @param LC View of a linked cell structure for finding neighbors quickly
 * @param shape shape as a sort of placeholder
 * @param n point where to center the placeholder
 * @return true - placeholder at node contains no points,
 *         false - there is at least one point contained
 */
bool IsVoidNode(const LinkedCell::LinkedCell_View &LC, const Shape &shape, const Node &n)
{
  LOG(1, "INFO: Node is " << n << ".");
  LOG(1, "INFO: Original shape is at " << shape.getCenter() << " with radius of "
      << shape.getRadius() << ".");
  const Shape shapeAtn = translate(shape, n);
  LOG(1, "INFO: Translated shape is at " << shapeAtn.getCenter() << ".");
  LinkedCell::LinkedList neighborlist = LC.getAllNeighbors(
      shapeAtn.getRadius(), shapeAtn.getCenter());
  LOG(1, "INFO: we have " << neighborlist.size() << " possible neighbors.");
  LinkedCell::LinkedList::const_iterator neighboriter =
      std::find_if(neighborlist.begin(), neighborlist.end(),
          boost::bind(&Shape::isInside, shapeAtn,
              boost::bind(&TesselPoint::getPosition, _1) ));
  return neighboriter == neighborlist.end();
}

FillPredicate IsVoidNode_FillPredicate(const Shape &_shape)
{
  // copy _shape !
  const LinkedCell::LinkedCell_View LC = World::getInstance().getLinkedCell(_shape.getRadius());
  AnyFillPredicate::FillPredicator function = boost::bind(&IsVoidNode, LC, _shape, _1);
  LOG(1, "INFO: Create IsVoidNode_FillPredicate from shape at "
      << _shape.getCenter() << " and radius " << _shape.getRadius() << ".");
  FillPredicate::impl_ptr impl(new AnyFillPredicate(function));
  return FillPredicate(impl);
}
