/*
 * bondgraph.cpp
 *
 *  Created on: Oct 29, 2009
 *      Author: heber
 */

#include <iostream>

#include "atom.hpp"
#include "bondgraph.hpp"
#include "element.hpp"
#include "molecule.hpp"
#include "parser.hpp"
#include "vector.hpp"

/** Constructor of class BondGraph.
 * This classes contains typical bond lengths and thus may be used to construct a bond graph for a given molecule.
 */
BondGraph::BondGraph(bool IsA) : BondLengthMatrix(NULL), IsAngstroem(IsA)
{
};

/** Destructor of class BondGraph.
 */
BondGraph::~BondGraph()
{
  if (BondLengthMatrix != NULL) {
    delete(BondLengthMatrix);
  }
};

/** Parses the bond lengths in a given file and puts them int a matrix form.
 * Allocates \a MatrixContainer and uses MatrixContainer::ParseMatrix().
 * \param *out output stream for debugging
 * \param filename file with bond lengths to parse
 * \return true - success in parsing file, false - failed to parse the file
 */
bool BondGraph::LoadBondLengthTable(ofstream * const out, const string &filename)
{
  bool status = true;

  // allocate MatrixContainer
  if (BondLengthMatrix != NULL) {
    *out << "MatrixContainer for Bond length already present, removing." << endl;
    delete(BondLengthMatrix);
  }
  BondLengthMatrix = new MatrixContainer;

  // parse in matrix
  BondLengthMatrix->ParseMatrix(filename.c_str(), 0, 1, 0);

  // find greatest distance
  max_distance=0;
  for(int i=0;i<BondLengthMatrix->RowCounter[0];i++)
    for(int j=i;j<BondLengthMatrix->ColumnCounter[0];j++)
      if (BondLengthMatrix->Matrix[0][i][j] > max_distance)
        max_distance = BondLengthMatrix->Matrix[0][i][j];

  return status;
};

/** Parses the bond lengths in a given file and puts them int a matrix form.
 * \param *out output stream for debugging
 * \param *mol molecule with atoms
 * \return true - success, false - failed to construct bond structure
 */
bool BondGraph::ConstructBondGraph(ofstream * const out, molecule * const mol)
{
  bool status = true;

  if (BondLengthMatrix == NULL)
   return false;
  mol->CreateAdjacencyList(out, max_distance, IsAngstroem, &BondGraph::BondLengthMatrixMinMaxDistance, this);

  return status;
};

/** Returns the entry for a given index pair.
 * \param firstelement index/atom number of first element (row index)
 * \param secondelement index/atom number of second element (column index)
 * \note matrix is of course symmetric.
 */
double BondGraph::GetBondLength(int firstZ, int secondZ)
{
  return (BondLengthMatrix->Matrix[0][firstZ][secondZ]);
};

/** Returns bond criterion for given pair based on covalent radius.
 * \param *Walker first BondedParticle
 * \param *OtherWalker second BondedParticle
 * \param &MinDistance lower bond bound on return
 * \param &MaxDistance upper bond bound on return
 * \param IsAngstroem whether units are in angstroem or bohr radii
 */
void BondGraph::CovalentMinMaxDistance(BondedParticle * const Walker, BondedParticle * const OtherWalker, double &MinDistance, double &MaxDistance, bool IsAngstroem)
{
  MinDistance = OtherWalker->type->CovalentRadius + Walker->type->CovalentRadius;
  MinDistance *= (IsAngstroem) ? 1. : 1. / AtomicLengthToAngstroem;
  MaxDistance = MinDistance + BONDTHRESHOLD;
  MinDistance -= BONDTHRESHOLD;
};

/** Returns bond criterion for given pair based on a bond length matrix.
 * The matrix should be contained in \a this BondGraph and contain an element-
 * to-element length.
 * \param *Walker first BondedParticle
 * \param *OtherWalker second BondedParticle
 * \param &MinDistance lower bond bound on return
 * \param &MaxDistance upper bond bound on return
 * \param IsAngstroem whether units are in angstroem or bohr radii
 */
void BondGraph::BondLengthMatrixMinMaxDistance(BondedParticle * const Walker, BondedParticle * const OtherWalker, double &MinDistance, double &MaxDistance, bool IsAngstroem)
{
  if (BondLengthMatrix->Matrix == NULL) {// safety measure if no matrix has been parsed yet
    cerr << Verbose(1) << "WARNING:  BondLengthMatrixMinMaxDistance() called without having parsed the bond length matrix yet!" << endl;
    CovalentMinMaxDistance(Walker, OtherWalker, MinDistance, MaxDistance, IsAngstroem);
  } else {
    MinDistance = GetBondLength(Walker->type->Z-1, OtherWalker->type->Z-1);
    MinDistance *= (IsAngstroem) ? 1. : 1. / AtomicLengthToAngstroem;
    MaxDistance = MinDistance + BONDTHRESHOLD;
    MinDistance -= BONDTHRESHOLD;
  }
};

