/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * FillVoidWithMoleculeAction.cpp
 *
 *  Created on: May 10, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "Atom/atom.hpp"
#include "CodePatterns/Verbose.hpp"
#include "Descriptors/MoleculeIdDescriptor.hpp"
#include "Descriptors/MoleculeOrderDescriptor.hpp"
#include "Graph/BondGraph.hpp"
#include "molecule.hpp"
#include "MoleculeListClass.hpp"
#include "Parser/FormatParserInterface.hpp"
#include "Parser/FormatParserStorage.hpp"
#include "Tesselation/boundary.hpp"
#include "World.hpp"


#include <algorithm>
#include <iostream>
#include <string>

#include "Actions/MoleculeAction/FillVoidWithMoleculeAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "FillVoidWithMoleculeAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
ActionState::ptr MoleculeFillVoidWithMoleculeAction::performCall() {
  if (!boost::filesystem::exists(params.fillername.get())) {
    ELOG(1, "File with filler molecule " << params.fillername.get() << " does not exist!");
    return Action::failure;
  }

  LOG(1, "INFO: Filling Box with water molecules, "
      << " minimum distance to molecules" << params.boundary.get()
      << ", random atom displacement " << params.RandAtomDisplacement.get()
      << ", random molecule displacement " << params.RandMoleculeDisplacement.get()
      << ", distances between fillers (" << params.distances.get()[0] << "," << params.distances.get()[1] << "," << params.distances.get()[2]
      << "), MinDistance " << params.MinDistance.get()
      << ", DoRotate " << params.DoRotate.get() << ".");
  // construct water molecule
  std::vector<molecule *> presentmolecules = World::getInstance().getAllMolecules();
//  LOG(0, presentmolecules.size() << " molecules initially are present.");
  std::string FilenameSuffix = params.fillername.get().string().substr(params.fillername.get().string().find_last_of('.')+1, params.fillername.get().string().length());
  ifstream input;
  input.open(params.fillername.get().string().c_str());
  ParserTypes type = FormatParserStorage::getInstance().getTypeFromSuffix(FilenameSuffix);
  FormatParserInterface &parser = FormatParserStorage::getInstance().get(type);
  parser.load(&input);

  // search the filler molecule that has been just parsed
  molecule *filler = World::getInstance().getMolecule(MoleculeByOrder(-1)); // get last molecule
  ASSERT(filler != NULL,
      "MoleculeFillVoidWithMoleculeAction::performCall() - no last molecule found, nothing parsed?.");
  filler->SetNameFromFilename(params.fillername.get().string().c_str());
  World::AtomComposite Set = filler->getAtomSet();
  World::getInstance().getBondGraph()->CreateAdjacency(Set);

  // call routine
  double distance[NDIM];
  for (int i=0;i<NDIM;i++)
    distance[i] = params.distances.get()[i];
  FillVoidWithMolecule(
      filler,
      *(World::getInstance().getConfig()),
      distance,
      params.boundary.get(),
      params.RandAtomDisplacement.get(),
      params.RandMoleculeDisplacement.get(),
      params.MinDistance.get(),
      params.DoRotate.get());

  // generate list of newly created molecules
  // (we can in general remove more quickly from a list than a vector)
  std::vector<molecule *> fillermolecules = World::getInstance().getAllMolecules();
//  LOG(0, fillermolecules.size() << " molecules are present.");
  std::list<molecule *> fillermolecules_list;
  std::copy( fillermolecules.begin(),  fillermolecules.end(), std::back_inserter( fillermolecules_list ));
//  LOG(0, fillermolecules_list.size() << " molecules have been copied.");
  for (std::vector<molecule *>::const_iterator iter = presentmolecules.begin();
      iter != presentmolecules.end();
      ++iter) {
    fillermolecules_list.remove(*iter);
  }
//  LOG(0, fillermolecules_list.size() << " molecules left after removal.");
  fillermolecules.clear();
  std::copy(fillermolecules_list.begin(), fillermolecules_list.end(), std::back_inserter( fillermolecules ));

//  LOG(0, fillermolecules.size() << " molecules have been inserted.");

  return ActionState::ptr(new MoleculeFillVoidWithMoleculeState(fillermolecules,params));
}

ActionState::ptr MoleculeFillVoidWithMoleculeAction::performUndo(ActionState::ptr _state) {
  MoleculeFillVoidWithMoleculeState *state = assert_cast<MoleculeFillVoidWithMoleculeState*>(_state.get());

  MoleculeListClass *MolList = World::getInstance().getMolecules();

  BOOST_FOREACH(molecule *_mol, state->fillermolecules) {
    MolList->erase(_mol);
    if ((_mol != NULL) && (!(World::getInstance().getAllMolecules(MoleculeById(_mol->getId()))).empty())) {
      for(molecule::iterator iter = _mol->begin();
          !_mol->empty();
          iter = _mol->begin()) {
        atom *Walker = *iter;
        World::getInstance().destroyAtom(Walker);
      }
      World::getInstance().destroyMolecule(_mol);
    }
  }

  // as molecules and atoms from state are removed, we have to create a new one
  std::vector<molecule *> fillermolecules;
  return ActionState::ptr(new MoleculeFillVoidWithMoleculeState(fillermolecules,state->params));
}

ActionState::ptr MoleculeFillVoidWithMoleculeAction::performRedo(ActionState::ptr _state){
  //MoleculeFillVoidWithMoleculeState *state = assert_cast<MoleculeFillVoidWithMoleculeState*>(_state.get());

  return Action::failure;
  //return ActionState::ptr(_state);
}

bool MoleculeFillVoidWithMoleculeAction::canUndo() {
  return true;
}

bool MoleculeFillVoidWithMoleculeAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
