/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * BondAddAction.cpp
 *
 *  Created on: Nov 12, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "Atom/atom.hpp"
#include "Bond/bond.hpp"
#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"
#include "Descriptors/AtomIdDescriptor.hpp"
#include "World.hpp"
#include "WorldTime.hpp"

#include <iostream>
#include <string>

#include "Actions/BondAction/BondAddAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "BondAddAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
ActionState::ptr BondAddAction::performCall() {
  // check preconditions
  if (World::getInstance().countSelectedAtoms() != 2) {
    ELOG(1, "Exactly two atoms must be selected for BondAction Add.");
    return Action::failure;
  }
  const std::vector<atom *> selected_atoms = World::getInstance().getSelectedAtoms();
  if (selected_atoms[0]->IsBondedTo(WorldTime::getTime(), selected_atoms[1])) {
    ELOG(2, "There already is a bond in between the two selected atoms.");
    return Action::failure;
  }

  // create undo
  BondAddState *UndoState = new BondAddState(selected_atoms[0]->getId(), selected_atoms[1]->getId(), params);

  // execute action
  selected_atoms[0]->addBond(WorldTime::getTime(), selected_atoms[1]);
  ASSERT( selected_atoms[0]->IsBondedTo(WorldTime::getTime(), selected_atoms[1]),
      "BondAddAction::performCall() - adding bond in between "
      +toString(*selected_atoms[0])+" and "+toString(*selected_atoms[1])+" failed.");

  return ActionState::ptr(UndoState);
}

ActionState::ptr BondAddAction::performUndo(ActionState::ptr _state) {
  BondAddState *state = assert_cast<BondAddState*>(_state.get());

  // check whether bond already existed
  atom * const first = World::getInstance().getAtom(AtomById(state->firstId));
  atom * const second = World::getInstance().getAtom(AtomById(state->secondId));
  ASSERT((first != NULL) && (second != NULL),
      "BondAddAction::performUndo() - at least one of the ids "
      +toString(state->firstId)+" or "+toString(state->secondId)+" is not present.");
  if (first->IsBondedTo(WorldTime::getTime(), second)) {
    first->removeBond(WorldTime::getTime(), second);
  } else {
    ELOG(2, "There is no bond in between "+toString(state->firstId)
        +" and "+toString(state->secondId)+".");
  }

  return ActionState::ptr(_state);
}

ActionState::ptr BondAddAction::performRedo(ActionState::ptr _state){
  BondAddState *state = assert_cast<BondAddState*>(_state.get());

  // check whether bond already existed
  atom * const first = World::getInstance().getAtom(AtomById(state->firstId));
  atom * const second = World::getInstance().getAtom(AtomById(state->secondId));
  ASSERT((first != NULL) && (second != NULL),
      "BondAddAction::performRedo() - at least one of the ids "
      +toString(state->firstId)+" or "+toString(state->secondId)+" is not present.");
  if (!first->IsBondedTo(WorldTime::getTime(), second)) {
    first->addBond(WorldTime::getTime(), second);
  } else {
    ELOG(2, "There is already a bond in between "+toString(state->firstId)
        +" and "+toString(state->secondId)+".");
  }

  return ActionState::ptr(_state);
}

bool BondAddAction::canUndo() {
  return true;
}

bool BondAddAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
