/**
 * @file   command_list.cpp
 * @author Julian Iseringhausen <isering@ins.uni-bonn.de>
 * @date   Tue Apr  5 20:15:06 2011
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef DEBUG_BARRIER
#ifdef HAVE_MPI
#include <mpi.h>
#ifdef HAVE_MARMOT
#include <enhancempicalls.h>
#include <sourceinfompicalls.h>
#endif
#endif
#endif

#include <cstdio>

#include "base/command.hpp"
#include "base/command_factory.hpp"
#include "base/command_list.hpp"
#include "base/timer.hpp"
#include "base/defs.hpp"
#include "mg.hpp"

using namespace VMG;

Request CommandList::ExecuteList()
{
  Request request;
  Request final_request = (commands.size() == 0 ? StopCycleNow : Continue);
  Comm* comm = MG::GetComm();

  for (CommandList::iterator iter=commands.begin(); iter!=commands.end(); ++iter) {

#ifdef DEBUG
    const int num_args = (iter->second.size() > 1 ? iter->second.size() : (iter->second[0] == "" ? 0 : 1));
    MG::GetCommands().CheckNumberOfArguments(iter->first, num_args);
#endif

#ifdef DEBUG_BARRIER
#ifdef HAVE_MPI
    MPI_Barrier(MPI_COMM_WORLD);
#endif
    comm->PrintStringOnce("Command \"%s\" start", iter->first.c_str());
#endif

    Timer::Start(iter->first);
    request = MG::GetCommands().Get(iter->first)->Run(iter->second);
    Timer::Stop(iter->first);

#ifdef DEBUG_BARRIER
#ifdef HAVE_MPI
    MPI_Barrier(MPI_COMM_WORLD);
#endif
    comm->PrintStringOnce("Command \"%s\" done", iter->first.c_str());
#endif

    if (request == StopCycleLater)
      final_request = StopCycleNow;
    else if (request == StopCycleNow) {
      final_request = StopCycleNow;
      break;
    }
  }

  return final_request;
}

void CommandList::AddCommand(std::string command, std::string arguments)
{
  std::vector<std::string> argument_list;
  size_t pos;

  do {
    pos = arguments.find(':');
    argument_list.push_back(arguments.substr(0, pos));
    arguments.erase(0, pos+1);
  }while (pos != std::string::npos);

  commands.push_back(std::pair<std::string, std::vector<std::string> >(command, argument_list));
}

void CommandList::DeleteCommand(const CommandList::iterator& iter)
{
  commands.erase(iter);
}

void CommandList::Print()
{
  for (CommandList::iterator iter=commands.begin(); iter!=commands.end(); ++iter)
    printf("%s\n", (*iter).first.c_str());
}

void CommandList::Clear()
{
  commands.clear();
}
