/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2014 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * FitCompoundPotentialAction.cpp
 *
 *  Created on: Sep 10, 2014
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// needs to come before MemDebug due to placement new
#include <boost/archive/text_iarchive.hpp>

#include "CodePatterns/MemDebug.hpp"

#include <algorithm>
#include <boost/filesystem.hpp>
#include <fstream>

#include "Actions/PotentialAction/FitCompoundPotentialAction.hpp"

#include "CodePatterns/Log.hpp"

#include "Fragmentation/Homology/HomologyContainer.hpp"
#include "Fragmentation/Homology/HomologyGraph.hpp"
#include "Fragmentation/Summation/SetValues/Fragment.hpp"
#include "Potentials/PotentialTrainer.hpp"
#include "Potentials/SerializablePotential.hpp"
#include "World.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "FitCompoundPotentialAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */

ActionState::ptr PotentialFitCompoundPotentialAction::performCall() {
  // get homologies container
  HomologyContainer &homologies = World::getInstance().getHomologies();

  // first we try to look into the HomologyContainer
  LOG(1, "INFO: Listing all present homologies ...");
  for (HomologyContainer::container_t::const_iterator iter =
      homologies.begin(); iter != homologies.end(); ++iter) {
    LOG(1, "INFO: graph " << iter->first
        << " has Fragment " << iter->second.fragment
        << ", associated energy " << iter->second.energy
        << ", and sampled grid integral " << iter->second.charge_distribution.integral()
        << ".");
  }

  // fragment specifies the homology fragment to use
  SerializablePotential::ParticleTypes_t fragmentnumbers =
      PotentialTrainer::getNumbersFromElements(params.fragment.get());

  // then we ought to pick the right HomologyGraph ...
  const HomologyGraph graph =
      PotentialTrainer::getFirstGraphwithSpecifiedElements(homologies,fragmentnumbers);
  if (graph != HomologyGraph()) {
    LOG(1, "First representative graph containing fragment "
        << fragmentnumbers << " is " << graph << ".");
  } else {
    STATUS("Specific fragment "+toString(fragmentnumbers)+" not found in homologies!");
    return Action::failure;
  }

  // next we use the set of homologous fragments for training from the present potentials
  PotentialTrainer trainer;
  const bool status = trainer(
      homologies,
      graph,
      params.training_file.get(),
      params.max_iterations.get(),
      params.threshold.get(),
      params.best_of_howmany.get());
  if (!status) {
    STATUS("No required parameter derivatives for a box constraint minimization known.");
    return Action::failure;
  }

  return Action::success;
}

ActionState::ptr PotentialFitCompoundPotentialAction::performUndo(ActionState::ptr _state) {
  return Action::success;
}

ActionState::ptr PotentialFitCompoundPotentialAction::performRedo(ActionState::ptr _state){
  return Action::success;
}

bool PotentialFitCompoundPotentialAction::canUndo() {
  return false;
}

bool PotentialFitCompoundPotentialAction::shouldUndo() {
  return false;
}
/** =========== end of function ====================== */
