/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * AtomicInfo.cpp
 *
 *  Created on: Aug 10, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "atom.hpp"
#include "AtomicInfo.hpp"
#include "CodePatterns/Assert.hpp"
#include "Descriptors/AtomIdDescriptor.hpp"
#include "Descriptors/MoleculeIdDescriptor.hpp"
#include "Element/element.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "molecule.hpp"
#include "World.hpp"

/********************************** Functions for class AtomicInfo **********************************/

AtomicInfo::AtomicInfo() :
    Type(NULL),
    charge(0.),
    FatherId(0),
    MolId(0),
    Id(0),
    Nr(0)
{}
AtomicInfo::AtomicInfo(const atom &_atom) :
    Position(_atom.getPosition()),
    Type(_atom.getType()),
    charge(_atom.getCharge()),
    Velocity(_atom.getAtomicVelocity()),
    FatherId(_atom.father->getId()),
    MolId(0),
    Id(_atom.getId()),
    Nr(_atom.getNr())
{
  const molecule * const mol = _atom.getMolecule();
  ASSERT(mol != NULL,
      "Atom "+toString(_atom.getId())+" is not associated with any molecule.");
  MolId = mol->getId();
}

AtomicInfo::~AtomicInfo()
{}


bool AtomicInfo::setAtom(atom &_atom) const
{
  _atom.setPosition(Position);
  _atom.setType(Type);
  _atom.setCharge(charge);
  _atom.setAtomicVelocity(Velocity);

  // set old id
  bool status = true;
  if (_atom.getId() != Id)
    status = _atom.changeId(Id);

  // set its father
  atom * const _father = World::getInstance().getAtom(AtomById(FatherId));
  if (_father == NULL)
    _atom.father = &_atom;
  else
    _atom.father = _father;

  // setting molecule
  molecule * const _mol = World::getInstance().getMolecule(MoleculeById(MolId));
  if (_mol != NULL)
    _atom.setMolecule(_mol); // this is ok, mol is const within AtomicInfo, but not outside (atoms need to register)
  _atom.changeNr(Nr);

  return status;
}

atomId_t AtomicInfo::getId() const {
  return Id;
}

AtomicInfo& AtomicInfo::operator=(const AtomicInfo& AI)
{
  if (&AI == this)  // check self-assignment
    return *this;
  Position = AI.Position;
  Type = AI.Type;
  FatherId = AI.FatherId;
  MolId = AI.MolId;
  Velocity = AI.Velocity;
  Id = AI.Id;
  Nr = AI.Nr;
  return *this;
}

