/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2014 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * SphericalPointDistributionUnitTest.cpp
 *
 *  Created on: May 29, 2014
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

// include headers that implement a archive in simple text format
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>

#include "SphericalPointDistributionUnitTest.hpp"

#include <boost/assign.hpp>
#include <boost/math/quaternion.hpp>

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"

#include "LinearAlgebra/Line.hpp"

#include "Fragmentation/Exporters/SphericalPointDistribution.hpp"

#include "LinearAlgebra/Line.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

using namespace boost::assign;

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( SphericalPointDistributionTest );


void SphericalPointDistributionTest::setUp()
{
  // failing asserts should be thrown
  ASSERT_DO(Assert::Throw);

  setVerbosity(5);
}


void SphericalPointDistributionTest::tearDown()
{
}


/** UnitTest for matchSphericalPointDistributions() with two points
 */
void SphericalPointDistributionTest::matchSphericalPointDistributionsTest_2()
{
  SphericalPointDistribution SPD(1.);
  // test with one point, matching trivially
  {
    SphericalPointDistribution::Polygon_t polygon;
    polygon += Vector(1.,0.,0.);
    SphericalPointDistribution::Polygon_t newpolygon =
        SPD.get<2>();
    SphericalPointDistribution::Polygon_t expected;
    expected += Vector(-1.,0.,0.);
    SphericalPointDistribution::Polygon_t remaining =
        SphericalPointDistribution::matchSphericalPointDistributions(
            polygon,
            newpolygon);
    CPPUNIT_ASSERT_EQUAL( expected, remaining );
  }

  // test with one point, just a flip of axis
  {
    SphericalPointDistribution::Polygon_t polygon;
    polygon += Vector(0.,1.,0.);
    SphericalPointDistribution::Polygon_t newpolygon =
        SPD.get<2>();
    SphericalPointDistribution::Polygon_t expected;
    expected += Vector(0.,-1.,0.);
    SphericalPointDistribution::Polygon_t remaining =
        SphericalPointDistribution::matchSphericalPointDistributions(
            polygon,
            newpolygon);
    CPPUNIT_ASSERT_EQUAL( expected, remaining );
  }

  // test with one point, just a flip to another axis
  {
    SphericalPointDistribution::Polygon_t polygon;
    polygon += Vector(0.,0.,-1.);
    SphericalPointDistribution::Polygon_t newpolygon =
        SPD.get<2>();
    SphericalPointDistribution::Polygon_t expected;
    expected += Vector(0.,0.,1.);
    SphericalPointDistribution::Polygon_t remaining =
        SphericalPointDistribution::matchSphericalPointDistributions(
            polygon,
            newpolygon);
    CPPUNIT_ASSERT_EQUAL( expected, remaining );
  }

  // test with one point, full rotation
  {
    Line RotationAxis(zeroVec, Vector(0.2, 0.43, 0.6893248));
    SphericalPointDistribution::Polygon_t polygon;
    polygon += RotationAxis.rotateVector(Vector(1.,0.,0.), 47.6/180*M_PI);
    SphericalPointDistribution::Polygon_t newpolygon =
        SPD.get<2>();
    SphericalPointDistribution::Polygon_t expected;
    expected += RotationAxis.rotateVector(Vector(-1.,0.,0.), 47.6/180*M_PI);
    SphericalPointDistribution::Polygon_t remaining =
        SphericalPointDistribution::matchSphericalPointDistributions(
            polygon,
            newpolygon);
    CPPUNIT_ASSERT_EQUAL( expected, remaining );
  }
}

/** UnitTest for matchSphericalPointDistributions() with three points
 */
void SphericalPointDistributionTest::matchSphericalPointDistributionsTest_3()
{
  SphericalPointDistribution SPD(1.);

  // test with one point, matching trivially
  {
    SphericalPointDistribution::Polygon_t polygon;
    polygon += Vector(1.,0.,0.);
    SphericalPointDistribution::Polygon_t newpolygon =
        SPD.get<3>();
    SphericalPointDistribution::Polygon_t expected = newpolygon;
    expected.pop_front(); // remove first point
    SphericalPointDistribution::Polygon_t remaining =
        SphericalPointDistribution::matchSphericalPointDistributions(
            polygon,
            newpolygon);
    CPPUNIT_ASSERT_EQUAL( expected, remaining );
  }

  // test with one point, just a flip of x and y axis
  {
    SphericalPointDistribution::Polygon_t polygon;
    polygon += Vector(0.,1.,0.);
    SphericalPointDistribution::Polygon_t newpolygon =
        SPD.get<3>();
    SphericalPointDistribution::Polygon_t expected = newpolygon;
    expected.pop_front(); // remove first point
    for (SphericalPointDistribution::Polygon_t::iterator iter = expected.begin();
        iter != expected.end(); ++iter) {
      std::swap((*iter)[0], (*iter)[1]);
      (*iter)[0] *= -1.;
    }
    SphericalPointDistribution::Polygon_t remaining =
        SphericalPointDistribution::matchSphericalPointDistributions(
            polygon,
            newpolygon);
    CPPUNIT_ASSERT_EQUAL( expected, remaining );
  }
}

/** UnitTest for matchSphericalPointDistributions() with four points
 */
void SphericalPointDistributionTest::matchSphericalPointDistributionsTest_4()
{
  SphericalPointDistribution SPD(1.);

  // test with one point, matching trivially
  {
    SphericalPointDistribution::Polygon_t polygon;
    polygon += Vector(1.,0.,0.);
    SphericalPointDistribution::Polygon_t newpolygon =
        SPD.get<4>();
    SphericalPointDistribution::Polygon_t expected = newpolygon;
    expected.pop_front(); // remove first point
    SphericalPointDistribution::Polygon_t remaining =
        SphericalPointDistribution::matchSphericalPointDistributions(
            polygon,
            newpolygon);
    CPPUNIT_ASSERT_EQUAL( expected, remaining );
  }

  // test with one point, just a flip of axis
  {
    SphericalPointDistribution::Polygon_t polygon;
    polygon += Vector(0.,1.,0.);
    SphericalPointDistribution::Polygon_t newpolygon =
        SPD.get<4>();
    SphericalPointDistribution::Polygon_t expected = newpolygon;
    expected.pop_front(); // remove first point
    for (SphericalPointDistribution::Polygon_t::iterator iter = expected.begin();
        iter != expected.end(); ++iter) {
      std::swap((*iter)[0], (*iter)[1]);
      (*iter)[0] *= -1.;
    }
    SphericalPointDistribution::Polygon_t remaining =
        SphericalPointDistribution::matchSphericalPointDistributions(
            polygon,
            newpolygon);
    CPPUNIT_ASSERT_EQUAL( expected, remaining );
  }
}

/** UnitTest for matchSphericalPointDistributions() with five points
 */
void SphericalPointDistributionTest::matchSphericalPointDistributionsTest_5()
{
  SphericalPointDistribution SPD(1.);

  // test with one point, matching trivially
  {
    SphericalPointDistribution::Polygon_t polygon;
    polygon += Vector(1.,0.,0.);
    SphericalPointDistribution::Polygon_t newpolygon =
        SPD.get<5>();
    SphericalPointDistribution::Polygon_t expected = newpolygon;
    expected.pop_front(); // remove first point
    SphericalPointDistribution::Polygon_t remaining =
        SphericalPointDistribution::matchSphericalPointDistributions(
            polygon,
            newpolygon);
    CPPUNIT_ASSERT_EQUAL( expected, remaining );
  }

  // test with one point, just a flip of axis
  {
    SphericalPointDistribution::Polygon_t polygon;
    polygon += Vector(0.,1.,0.);
    SphericalPointDistribution::Polygon_t newpolygon =
        SPD.get<5>();
    SphericalPointDistribution::Polygon_t expected = newpolygon;
    expected.pop_front(); // remove first point
    for (SphericalPointDistribution::Polygon_t::iterator iter = expected.begin();
        iter != expected.end(); ++iter) {
      std::swap((*iter)[0], (*iter)[1]);
      (*iter)[0] *= -1.;
    }
    SphericalPointDistribution::Polygon_t remaining =
        SphericalPointDistribution::matchSphericalPointDistributions(
            polygon,
            newpolygon);
    CPPUNIT_ASSERT_EQUAL( expected, remaining );
  }
}

/** UnitTest for matchSphericalPointDistributions() with six points
 */
void SphericalPointDistributionTest::matchSphericalPointDistributionsTest_6()
{
  SphericalPointDistribution SPD(1.);

  // test with one point, matching trivially
  {
    SphericalPointDistribution::Polygon_t polygon;
    polygon += Vector(1.,0.,0.);
    SphericalPointDistribution::Polygon_t newpolygon =
        SPD.get<6>();
    SphericalPointDistribution::Polygon_t expected = newpolygon;
    expected.pop_front(); // remove first point
    SphericalPointDistribution::Polygon_t remaining =
        SphericalPointDistribution::matchSphericalPointDistributions(
            polygon,
            newpolygon);
    CPPUNIT_ASSERT_EQUAL( expected, remaining );
  }

  // test with one point, just a flip of axis
  {
    SphericalPointDistribution::Polygon_t polygon;
    polygon += Vector(0.,1.,0.);
    SphericalPointDistribution::Polygon_t newpolygon =
        SPD.get<6>();
    SphericalPointDistribution::Polygon_t expected = newpolygon;
    expected.pop_front(); // remove first point
    for (SphericalPointDistribution::Polygon_t::iterator iter = expected.begin();
        iter != expected.end(); ++iter) {
      std::swap((*iter)[0], (*iter)[1]);
      (*iter)[0] *= -1.;
    }
    SphericalPointDistribution::Polygon_t remaining =
        SphericalPointDistribution::matchSphericalPointDistributions(
            polygon,
            newpolygon);
    CPPUNIT_ASSERT_EQUAL( expected, remaining );
  }
}

/** UnitTest for matchSphericalPointDistributions() with seven points
 */
void SphericalPointDistributionTest::matchSphericalPointDistributionsTest_7()
{
  SphericalPointDistribution SPD(1.);

  // test with one point, matching trivially
  {
    SphericalPointDistribution::Polygon_t polygon;
    polygon += Vector(1.,0.,0.);
    SphericalPointDistribution::Polygon_t newpolygon =
        SPD.get<7>();
    SphericalPointDistribution::Polygon_t expected = newpolygon;
    expected.pop_front(); // remove first point
    SphericalPointDistribution::Polygon_t remaining =
        SphericalPointDistribution::matchSphericalPointDistributions(
            polygon,
            newpolygon);
    CPPUNIT_ASSERT_EQUAL( expected, remaining );
  }

  // test with one point, just a flip of axis
  {
    SphericalPointDistribution::Polygon_t polygon;
    polygon += Vector(0.,1.,0.);
    SphericalPointDistribution::Polygon_t newpolygon =
        SPD.get<7>();
    SphericalPointDistribution::Polygon_t expected = newpolygon;
    expected.pop_front(); // remove first point
    for (SphericalPointDistribution::Polygon_t::iterator iter = expected.begin();
        iter != expected.end(); ++iter) {
      std::swap((*iter)[0], (*iter)[1]);
      (*iter)[0] *= -1.;
    }
    SphericalPointDistribution::Polygon_t remaining =
        SphericalPointDistribution::matchSphericalPointDistributions(
            polygon,
            newpolygon);
    CPPUNIT_ASSERT_EQUAL( expected, remaining );
  }
}

/** UnitTest for matchSphericalPointDistributions() with eight points
 */
void SphericalPointDistributionTest::matchSphericalPointDistributionsTest_8()
{
  SphericalPointDistribution SPD(1.);

  // test with one point, matching trivially
  {
    SphericalPointDistribution::Polygon_t polygon;
    polygon += Vector(1.,0.,0.);
    SphericalPointDistribution::Polygon_t newpolygon =
        SPD.get<8>();
    SphericalPointDistribution::Polygon_t expected = newpolygon;
    expected.pop_front(); // remove first point
    SphericalPointDistribution::Polygon_t remaining =
        SphericalPointDistribution::matchSphericalPointDistributions(
            polygon,
            newpolygon);
    CPPUNIT_ASSERT_EQUAL( expected, remaining );
  }

  // test with one point, just a flip of axis
  {
    SphericalPointDistribution::Polygon_t polygon;
    polygon += Vector(0.,1.,0.);
    SphericalPointDistribution::Polygon_t newpolygon =
        SPD.get<8>();
    SphericalPointDistribution::Polygon_t expected = newpolygon;
    expected.pop_front(); // remove first point
    for (SphericalPointDistribution::Polygon_t::iterator iter = expected.begin();
        iter != expected.end(); ++iter) {
      std::swap((*iter)[0], (*iter)[1]);
      (*iter)[0] *= -1.;
    }
   SphericalPointDistribution::Polygon_t remaining =
        SphericalPointDistribution::matchSphericalPointDistributions(
            polygon,
            newpolygon);
    CPPUNIT_ASSERT_EQUAL( expected, remaining );
  }
}
