/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2014 Frederik Heber. All rights reserved.
 *
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * QtFavoriteActions.cpp
 *
 *  Created on: Sep 11, 2014
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <Qt/qaction.h>
#include <Qt/qpainter.h>
#include <Qt/qpixmap.h>
#include <Qt/qsettings.h>

#include "UIElements/Views/Qt4/QtToolBar.hpp"

#include "CodePatterns/MemDebug.hpp"

#include <string>

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"

#include "Actions/Action.hpp"
#include "Actions/ActionQueue.hpp"
#include "Actions/ActionRegistry.hpp"
#include "Actions/ActionTraits.hpp"
#include "Actions/OptionTrait.hpp"

using namespace MoleCuilder;

const Observable::channels_t
QtToolBar::QtFavoriteActions::ActionQueuedChannels(1, ActionQueue::ActionQueued);

QtToolBar::QtFavoriteActions::QtFavoriteActions() :
  Observer("QtFavoriteActions"),
  LastActionCalled(
      ActionQueue::getPointer(),
      boost::bind(&QtToolBar::QtFavoriteActions::updateQueuedAction, boost::cref(this)),
      "QtFavoriteActions_QueuedAction",
      std::string(""),
      ActionQueuedChannels),
  ActionQueue_observing(false)
{
  // initialize counts from settings
  QSettings settings;
  settings.beginGroup("FavoriteActions");
  const ActionQueue::ActionTokens_t actions = ActionQueue::getInstance().getListOfActions();
  for (ActionQueue::ActionTokens_t::const_iterator tokeniter = actions.begin();
      tokeniter != actions.end(); ++tokeniter) {
#ifndef NDEBUG
    std::pair<ActionCounts_t::iterator, bool> inserter =
#endif
    ActionCounts.insert( std::make_pair(
        *tokeniter,
        settings.value(QString(tokeniter->c_str()), 0).toInt())
    );
    ASSERT( inserter.second,
        "QtFavoriteActions::QtFavoriteActions() - encountered action token"
        +*tokeniter+" twice.");
  }

//  resize(settings.value("size", QSize(400, 400)).toSize());
  settings.endGroup();

  // sign in
  ActionQueue::getInstance().signOn(this, ActionQueue::ActionQueued);
  ActionQueue_observing = true;
}

std::string QtToolBar::QtFavoriteActions::updateQueuedAction() const
{
  const Action *lastcalledaction =
      ActionQueue::getInstance().lastChanged<Action>();
  if (lastcalledaction != NULL)
    return lastcalledaction->getName();
  else
    return std::string("");
}

void QtToolBar::QtFavoriteActions::resetQueuedAction()
{
  const std::string name = LastActionCalled.get();
  if (!name.empty())
    ++ActionCounts[name];
}

QtToolBar::QtFavoriteActions::~QtFavoriteActions()
{
  // sign off
  if (ActionQueue_observing)
    ActionQueue::getInstance().signOff(this, ActionQueue::ActionQueued);

  // store all known counts
  QSettings settings;
  settings.beginGroup("FavoriteActions");
  ActionQueue::ActionTokens_t actions = ActionQueue::getInstance().getListOfActions();
  for (ActionQueue::ActionTokens_t::const_iterator tokeniter = actions.begin();
      tokeniter != actions.end(); ++tokeniter) {
    settings.setValue(QString(tokeniter->c_str()), ActionCounts[*tokeniter]);
  }
  settings.endGroup();
}

void QtToolBar::QtFavoriteActions::update(Observable *publisher)
{
  ASSERT(0, "QtFavoriteActions::update() - this should never be called, we are only subscribed to channels.");
}

void QtToolBar::QtFavoriteActions::subjectKilled(Observable *publisher)
{
  ActionQueue_observing = false;
}

void QtToolBar::QtFavoriteActions::recieveNotification(Observable *publisher, Notification_ptr notification)
{
  if (dynamic_cast<ActionQueue *>(publisher) != NULL) {
    resetQueuedAction();
  } else {
    ASSERT(0, "QtFavoriteActions::update() - cannot get here, we are only subscribed to ActionQueue.");
  }
}

void
QtToolBar::QtFavoriteActions::addToolBarActions(
    QtToolBar &_toolbar,
    const unsigned int _max) const
{
  // invert map
  std::multimap<unsigned int, std::string> InvertedCounts;
  for (ActionCounts_t::const_iterator countiter = ActionCounts.begin();
      countiter != ActionCounts.end(); ++countiter) {
    InvertedCounts.insert( std::make_pair( countiter->second, countiter->first) );
  }

  // then use last _max entries
  std::multimap<unsigned int, std::string>::const_reverse_iterator iter =
      InvertedCounts.rbegin();
  ActionQueue &AQ = ActionQueue::getInstance();
  const present_actions_t &present_actions = _toolbar.getPresentActions();
  for (size_t added = 0; added<_max; ++iter) {
    const std::string &token = iter->second;
    present_actions_t::const_iterator actioniter =
        present_actions.find(token);
    if (actioniter == present_actions.end()) {
      const ActionTrait &traits = AQ.getActionsTrait(token);
      const std::string icon_name = traits.getMenuName() + std::string("-") + token;
      const std::string description = traits.getDescription();
      QIcon icon = getIcon(token, icon_name);
      _toolbar.addActionItem(
          token,
          description,
          icon_name);
      ++added;
    }
  }
}

QIcon
QtToolBar::QtFavoriteActions::createIconPlaceholder(
    const std::string &_token
    ) const
{
  LOG(2, "DEBUG: Creating icon with text " << _token);
  QPixmap pixmap(100,100);
  pixmap.fill(QColor("white"));

  QPainter painter( &pixmap  );
  QRect rectangle( QPoint(0,0), QPoint(100,100));
  QRect boundingbox = QRect(1,1, 98, 98 );
  painter.drawRect( boundingbox );
  QFont font = QFont("Arial");
  font.setPixelSize(30);
  painter.setFont( font );
  QString icon_text(_token.c_str());
  icon_text.replace(QString("-"), QString("\n"));
  painter.drawText( rectangle, Qt::TextWordWrap, icon_text, &boundingbox );

  QIcon PlaceholderIcon(pixmap);

  return PlaceholderIcon;
}

QIcon
QtToolBar::QtFavoriteActions::getIcon(
    const std::string &_token,
    const std::string &_icon_name
    ) const
{
  LOG(2, "DEBUG: Obtaining icon with text " << _token);
  // TODO: look up _icon_name from QIcon::fromTheme or someplace
  if (QIcon::hasThemeIcon(QString(_icon_name.c_str())))
    return QIcon::fromTheme(QString(_icon_name.c_str()));
  else
    return createIconPlaceholder(_token);
}
