/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * OptionRegistry.cpp
 *
 *  Created on: Oct 26, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include <boost/shared_ptr.hpp>

#include "Actions/Action.hpp"
#include "Actions/ActionRegistry.hpp"
#include "Actions/ActionTraits.hpp"
#include "Actions/OptionRegistry.hpp"
#include "Actions/OptionTrait.hpp"
#include "CodePatterns/Singleton_impl.hpp"
#include "CodePatterns/Registry_impl.hpp"

/** Constructor for class OptionRegistry.
 */
OptionRegistry::OptionRegistry()
{}

/** Destructor for class OptionRegistry.
 */
OptionRegistry::~OptionRegistry()
{}

/** Just passes on call to Registry<std::type_info>::getByName().
 * \param name name of Option
 * \return pointer to Option's type
 */
const OptionTrait* OptionRegistry::getOptionByName(const std::string name) const
{
  return const_cast<const OptionTrait *>(getByName(name));
}

/** Initializes options from ActionRegistries contents.
 * Just goes through all Action's and adds their options.
 */
//void OptionRegistry::Init()
//{
//  ActionRegistry &AR = ActionRegistry::getInstance();
//  for (ActionRegistry::const_iterator actioniter = AR.getBeginIter();
//      actioniter != AR.getEndIter();
//      ++actioniter) {
//    Action* const currentAction = actioniter->second;
//    for (ActionTraits::options_const_iterator optioniter = currentAction->Traits.getBeginIter();
//        optioniter != currentAction->Traits.getEndIter();
//        ++optioniter) {
//      // wrap option in a shared_ptr to have it exist till and automatically removed at end of code
//      std::string const &OptionName = optioniter->first;
//      std::type_info const &typeDesired = *((actioniter->second)->Traits.getOptionType(OptionName));
//      if (isOptionPresentByName(OptionName)) {
//        std::type_info const &typePresent = *(getOptionByName(OptionName)->getType());
//        ASSERT( typeDesired == typePresent,
//            "OptionRegistry::Init() - Two options with same token do not have the same type!");
//      } else {
//        registerInstance(new Option(OptionName, &typeDesired));
//      }
//    }
//  }
//}

/** checking that each of the Actions' options has the same (and present) type in the OptionRegistry.
 * We use ASSERT
 */
void OptionRegistry::ConsistencyCheck()
{
  ActionRegistry &AR = ActionRegistry::getInstance();
  for (ActionRegistry::const_iterator actioniter = AR.getBeginIter();
      actioniter != AR.getEndIter();
      ++actioniter) {
    for (ActionTraits::options_const_iterator optioniter = (actioniter->second)->Traits.getBeginIter();
        optioniter != (actioniter->second)->Traits.getEndIter();
        ++optioniter) {
#ifndef NDEBUG
      std::string const &OptionName = optioniter->first;
      std::type_info const &typeDesired = *((actioniter->second)->Traits.getOption(OptionName).getType());
      std::type_info const &typePresent = *(getOptionByName(OptionName)->getType());
#endif
      ASSERT( isOptionPresentByName(OptionName),
          "ConsistencyCheck() - option token "+OptionName+" not contained in OptionRegistry");
      ASSERT( typeDesired == typePresent,
          "ConsistencyCheck() - Two options -- "+toString(typeDesired.name())
          +" and "+toString(typePresent.name())+" -- with same token do not have the same type!");
    }
  }
};


/** Just passes on call to Registry<std::type_info>::isPresentByName().
 * \param name name of Option
 * \return true - Option instance present, false - not
 */
bool OptionRegistry::isOptionPresentByName(const std::string name) const
{
  return isPresentByName(name);
}

CONSTRUCT_SINGLETON(OptionRegistry)
CONSTRUCT_REGISTRY(OptionTrait)
