/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * SurfaceCorrelationAction.cpp
 *
 *  Created on: May 9, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "analysis_correlation.hpp"
#include "boundary.hpp"
#include "linkedcell.hpp"
#include "CodePatterns/Verbose.hpp"
#include "CodePatterns/Log.hpp"
#include "element.hpp"
#include "molecule.hpp"
#include "periodentafel.hpp"
#include "tesselation.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "World.hpp"

#include <iostream>
#include <string>

using namespace std;

#include "Actions/AnalysisAction/SurfaceCorrelationAction.hpp"

// and construct the stuff
#include "SurfaceCorrelationAction.def"
#include "Action_impl_pre.hpp"

/** =========== define the function ====================== */
Action::state_ptr AnalysisSurfaceCorrelationAction::performCall() {
  int ranges[3] = {1, 1, 1};
  ofstream output;
  ofstream binoutput;
  string type;
  BinPairMap *binmap = NULL;

  // obtain information
  getParametersfromValueStorage();

  // execute action
  output.open(params.outputname.string().c_str());
  binoutput.open(params.binoutputname.string().c_str());
  ASSERT(params.Boundary != NULL, "No molecule specified for SurfaceCorrelation.");
  const double radius = 4.;
  double LCWidth = 20.;
  if (params.BinEnd > 0) {
    if (params.BinEnd > 2.*radius)
      LCWidth = params.BinEnd;
    else
      LCWidth = 2.*radius;
  }

  // get the boundary
  class Tesselation *TesselStruct = NULL;
  const LinkedCell *LCList = NULL;
  // find biggest molecule
  std::vector<molecule*> molecules = World::getInstance().getSelectedMolecules();
  std::cout << "There are " << molecules.size() << " selected molecules." << std::endl;
  LCList = new LinkedCell(*params.Boundary, LCWidth);
  FindNonConvexBorder(params.Boundary, TesselStruct, LCList, radius, NULL);
  CorrelationToSurfaceMap *surfacemap = NULL;
  if (params.periodic)
    surfacemap = PeriodicCorrelationToSurface( molecules, params.elements, TesselStruct, LCList, ranges);
  else
    surfacemap = CorrelationToSurface( molecules, params.elements, TesselStruct, LCList);
  delete LCList;
  OutputCorrelationMap<CorrelationToSurfaceMap>(&output, surfacemap, OutputCorrelationToSurface_Header, OutputCorrelationToSurface_Value);
  // check whether radius was appropriate
  {
    double start; double end;
    GetMinMax( surfacemap, start, end);
    if (LCWidth < end)
      DoeLog(1) && (eLog()<< Verbose(1) << "Linked Cell width is smaller than the found range of values! Bins can only be correct up to: " << radius << "." << endl);
  }
  binmap = BinData( surfacemap, params.BinWidth, params.BinStart, params.BinEnd );
  OutputCorrelationMap<BinPairMap> ( &binoutput, binmap, OutputCorrelation_Header, OutputCorrelation_Value );
  delete TesselStruct;  // surfacemap contains refs to triangles! delete here, not earlier!
  delete(binmap);
  delete(surfacemap);
  output.close();
  binoutput.close();
  return Action::success;
}

Action::state_ptr AnalysisSurfaceCorrelationAction::performUndo(Action::state_ptr _state) {
  return Action::success;
}

Action::state_ptr AnalysisSurfaceCorrelationAction::performRedo(Action::state_ptr _state){
  return Action::success;
}

bool AnalysisSurfaceCorrelationAction::canUndo() {
  return true;
}

bool AnalysisSurfaceCorrelationAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
