/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * QtWorldView.cpp
 *
 *  Created on: Jan 21, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Views/Qt4/QtWorldView.hpp"

#include <iostream>

#include "CodePatterns/MemDebug.hpp"

#include "Atom/atom.hpp"
#include "Formula.hpp"
#include "molecule.hpp"
#include "MoleculeListClass.hpp"
#include "Actions/SelectionAction/Molecules/MoleculeByIdAction.hpp"
#include "Actions/SelectionAction/Molecules/NotMoleculeByIdAction.hpp"

using namespace std;

// maybe this should go with the definition of molecules

// some attributes need to be easier to find for molecules
// these attributes are skiped so far
const int QtWorldView::COLUMNCOUNT = COLUMNTYPES_MAX;
const char *QtWorldView::COLUMNNAMES[QtWorldView::COLUMNCOUNT]={"Name","Atoms","Formula","Occurrence"/*,"Size"*/};

QtWorldView::QtWorldView(QWidget * _parent) :
  QTreeWidget (_parent),
  Observer("QtWorldView")
{
  setColumnCount(COLUMNCOUNT);
  setSelectionMode(QAbstractItemView::MultiSelection);

  QStringList header;
  for(int i=0; i<COLUMNCOUNT;++i)
    header << COLUMNNAMES[i];
  setHeaderLabels(header);

  molecules = World::getInstance().getMolecules();
  molecules->signOn(this);
  update(molecules);

  //connect(this,SIGNAL(cellChanged(int,int)),this,SLOT(moleculeChanged(int,int)));
  connect(this,SIGNAL(itemSelectionChanged()),this,SLOT(rowSelected()));

}

QtWorldView::~QtWorldView()
{
  molecules->signOff(this);
}

void QtWorldView::update(Observable *publisher) {
  int numMolecules = molecules->ListOfMolecules.size();
  clear();
  molSelection.resize(numMolecules);

  // list of (unique) formulas in the world
  std::vector<Formula> formula;

  int i;
  MoleculeList::iterator iter;
  for(iter = molecules->ListOfMolecules.begin(),i=0;
      iter != molecules->ListOfMolecules.end();
      ++i,++iter) {

    // find group if already in list
    QTreeWidgetItem *groupItem = NULL;
    for (unsigned int j=0;j<formula.size();j++)
      if ((*iter)->getFormula() == formula[j]){
        groupItem = topLevelItem(j);
        break;
      }

    // new molecule type -> create new group
    if (!groupItem){
      formula.push_back((*iter)->getFormula());
      groupItem = new QTreeWidgetItem(this);
      groupItem->setText(0, QString((*iter)->getName().c_str()));
      groupItem->setText(1, QString::number((*iter)->getAtomCount()));
      groupItem->setText(2, QString((*iter)->getFormula().toString().c_str()));
      groupItem->setText(3, "0");
    }

    // add molecule
    QTreeWidgetItem *molItem = new QTreeWidgetItem(groupItem);
    molItem->setText(0, QString((*iter)->getName().c_str()));
    molItem->setText(1, QString::number((*iter)->getAtomCount()));
    molItem->setText(2, QString((*iter)->getFormula().toString().c_str()));
    const int index = (*iter)->IndexNr;
    molItem->setData(0, Qt::UserRole, QVariant(index));
    molItem->setSelected(World::getInstance().isSelected(*iter));


    // increase group occurrence
    int count = groupItem->text(3).toInt() + 1;
    groupItem->setText(3, QString::number(count));

    //molSelection[i]=nameWidget->isSelected();
  }
}

void QtWorldView::subjectKilled(Observable *publisher) {
}

void QtWorldView::moleculeChanged() {
  /*int idx = verticalHeaderItem(row)->data(Qt::UserRole).toInt();
  molecule *mol = molecules->ReturnIndex(idx);
  string cellValue = item(row,NAME)->text().toStdString();
  if(mol->getName() != cellValue && cellValue !="") {
    mol->setName(cellValue);
  }
  else if(cellValue==""){
    item(row,NAME)->setText(QString(mol->getName().c_str()));
  }*/
}


void QtWorldView::rowSelected(){

  // lookup all molecules in the treeWidget
  for (int i=0;i<topLevelItemCount();i++){
    QTreeWidgetItem *top = topLevelItem(i);
    for (int j=0;j<top->childCount();j++){

      // molecules are 1 level below the top
      QTreeWidgetItem *molItem = top->child(j);

      // molecule index stored as user data
      int index = molItem->data(0, Qt::UserRole).toInt();
      molecule *mol = molecules->ReturnIndex(index);
      ASSERT(mol, "QtWorldView::rowSelected()");

      // selection changed by user?
      bool molSelectedWorld = World::getInstance().isSelected(mol);
      bool molSelectedList = molItem->isSelected();
      //std::cout << molSelectedWorld << " " << molSelectedList << std::endl;

      if (molSelectedWorld != molSelectedList){

        // apply new selection state
        if (molSelectedList){
          //std::cout << "select molecule" << std::endl;
          MoleCuilder::SelectionMoleculeById(mol->getId());
        }else{
          //std::cout << "unselect molecule" << std::endl;
          MoleCuilder::SelectionNotMoleculeById(mol->getId());
        }
      }
    }
  }
}
