/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2011 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * \file FragmentScheduler.cpp
 *
 * This file strongly follows the Serialization example from the boost::asio
 * library (see server.cpp)
 *
 *  Created on: Oct 19, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// boost asio needs specific operator new
#include <boost/asio.hpp>

#include "CodePatterns/MemDebug.hpp"

#include <boost/bind.hpp>
#include <boost/lexical_cast.hpp>
#include <iostream>
#include <vector>
#include "Connection.hpp" // Must come before boost/serialization headers.
#include <boost/serialization/vector.hpp>
#include "CodePatterns/Info.hpp"
#include "CodePatterns/Log.hpp"
#include "FragmentJob.hpp"

#include "FragmentScheduler.hpp"

FragmentScheduler::FragmentScheduler(boost::asio::io_service& io_service, unsigned short port) :
  acceptor_(io_service,
      boost::asio::ip::tcp::endpoint(boost::asio::ip::tcp::v4(), port)
  )
{
  Info info(__FUNCTION__);
  FragmentJob s(std::string("test"), 1);
  JobsQueue.pushJob(s);

  // Start an accept operation for a new connection.
  connection_ptr new_conn(new Connection(acceptor_.get_io_service()));
  acceptor_.async_accept(new_conn->socket(),
    boost::bind(&FragmentScheduler::handle_accept, this,
      boost::asio::placeholders::error, new_conn));
}

/// Handle completion of a accept operation.
void FragmentScheduler::handle_accept(const boost::system::error_code& e, connection_ptr conn)
{
  Info info(__FUNCTION__);
  std::cout << "handle_accept called." << std::endl;
  if (!e)
  {
    // Successfully accepted a new connection.
    // Check whether there are jobs in the queue
    if (JobsQueue.isJobPresent()) {
      // pop a job and send it to the client.
      FragmentJob s(JobsQueue.popJob());
      // The connection::async_write() function will automatically
      // serialize the data structure for us.
      conn->async_write(s,
        boost::bind(&FragmentScheduler::handle_write, this,
        boost::asio::placeholders::error, conn));
    } else {
      ELOG(2, "There is currently no job present in the queue.");
    }

    // Start an accept operation for a new Connection.
    connection_ptr new_conn(new Connection(acceptor_.get_io_service()));
    acceptor_.async_accept(new_conn->socket(),
      boost::bind(&FragmentScheduler::handle_accept, this,
      boost::asio::placeholders::error, new_conn));
  }
  else
  {
    // An error occurred. Log it and return. Since we are not starting a new
    // accept operation the io_service will run out of work to do and the
    // server will exit.
    ELOG(0, e.message());
  }
}

/// Handle completion of a write operation.
void FragmentScheduler::handle_write(const boost::system::error_code& e, connection_ptr conn)
{
    Info info(__FUNCTION__);
  // Nothing to do. The socket will be closed automatically when the last
  // reference to the connection object goes away.
}

