/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2011 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * FragmentQueue.cpp
 *
 *  Created on: Oct 19, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "FragmentQueue.hpp"

#include "CodePatterns/Assert.hpp"

FragmentResult FragmentQueue::NoResult(-1);
FragmentResult FragmentQueue::ResultDelivered(-2);

/** Constructor for class FragmentQueue.
 *
 */
FragmentQueue::FragmentQueue()
{}

/** Destructor for class FragmentQueue.
 *
 */
FragmentQueue::~FragmentQueue()
{}

/** Pushes a FragmentJob into the internal queue for delivery to server.
 *
 * \note we throw assertion when jobid has already been used.
 *
 * \param job job to enter into queue
 */
void FragmentQueue::pushJob(FragmentJob &job)
{
  ASSERT(job.getId() != (JobId_t)-1,
      "FragmentQueue::pushJob() - job to push has no valid id.");
  ASSERT(!results.count(job.getId()),
      "FragmentQueue::pushJob() - job id "+toString(job.getId())+" has already been used.");
  jobs.push_back(job);
  results.insert( std::make_pair(job.getId(), NoResult));
}

/** Queries whether a job has already been finished and the result is present.
 *
 * \param jobid id of job to query
 * \return true - result is present, false - result is not present
 */
bool FragmentQueue::isResultPresent(JobId_t jobid) const
{
  ResultMap::const_iterator iter = results.find(jobid);
  return ((iter != results.end())
      && (iter->second != NoResult)
      && (iter->second != ResultDelivered));
}

/** Delivers result for a finished job.
 *
 * \note we throw assertion if not present
 *
 * \param jobid id of job
 * \return result for job of given \a jobid
 */
FragmentResult FragmentQueue::getResult(JobId_t jobid)
{
  ResultMap::iterator iter = results.find(jobid);
  ASSERT(iter != results.end(),
      "FragmentQueue::pushResult() - job "+toString(jobid)+" is not known to us.");
  ASSERT(iter->second != NoResult,
      "FragmentQueue::pushResult() - job "+toString(jobid)+"'s result has not arrived yet.");
  ASSERT(iter->second != ResultDelivered,
      "FragmentQueue::pushResult() - job "+toString(jobid)+"'s result has already been delivered.");
  /// store result
  FragmentResult result = iter->second;
  /// mark as delivered in map
  iter->second = ResultDelivered;
  /// and return result
  return result;
}

/** Pushes a result for a finished job.
 *
 * \note we throw assertion if job already has result or is not known.
 *
 * \param result result of job to store
 */
void FragmentQueue::pushResult(FragmentResult &result)
{
  /// check for presence
  ResultMap::iterator iter = results.find(result.getId());
  ASSERT(iter != results.end(),
      "FragmentQueue::pushResult() - job "+toString(result.getId())+" is not known to us.");
  ASSERT(iter->second == NoResult,
      "FragmentQueue::pushResult() - is not waiting for the result of job "+toString(result.getId())+".");
  /// and overwrite NoResult in found entry
  iter->second = result;
}
