/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * ThermoStatContainer.cpp
 *
 *  Created on: 12.06.2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Helpers/MemDebug.hpp"

#include <cstring>

#include "ConfigFileBuffer.hpp"
#include "config.hpp"
#include "Helpers/Log.hpp"
#include "ThermoStatContainer.hpp"
#include "Helpers/Verbose.hpp"

#include <Thermostats/Berendsen.hpp>
#include <Thermostats/GaussianThermostat.hpp>
#include <Thermostats/Langevin.hpp>
#include <Thermostats/NoseHoover.hpp>
#include <Thermostats/NoThermostat.hpp>
#include <Thermostats/Woodcock.hpp>

/** Constructor for class ThermoStatContainer.
 *
 */
ThermoStatContainer::ThermoStatContainer() :
  activeThermostat(0),
  TargetTemp(0.00095004455)
{
  ThermostatTraits<Thermostat> *BerendsenTrait = new ThermostatTraits<Berendsen>();
  availThermostats[BerendsenTrait->getName()] = BerendsenTrait;
  ThermostatTraits<Thermostat> *GaussianTrait = new ThermostatTraits<GaussianThermostat>();
  availThermostats[GaussianTrait->getName()] = GaussianTrait;
  ThermostatTraits<Thermostat> *LangevinTrait = new ThermostatTraits<Langevin>();
  availThermostats[LangevinTrait->getName()] = LangevinTrait;
  ThermostatTraits<Thermostat> *NoseHooverTrait = new ThermostatTraits<NoseHoover>();
  availThermostats[NoseHooverTrait->getName()] = NoseHooverTrait;
  ThermostatTraits<Thermostat> *NoThermostatTrait = new ThermostatTraits<NoThermostat>();
  availThermostats[NoThermostatTrait->getName()] = NoThermostatTrait;
  ThermostatTraits<Thermostat> *WoodcockTrait = new ThermostatTraits<Woodcock>();
  availThermostats[WoodcockTrait->getName()] = WoodcockTrait;

  // for debugging: list all thermostats
//  cout << "List of known thermostats: ";
//  for(traitsMap::iterator iter = availThermostats.begin();iter!=availThermostats.end();++iter){
//    cout << iter->first << " ";
//  }
//  cout << endl;

  ASSERT(availThermostats.size()==6,"Not all implemented thermostats referenced!\nDid you check the names in the traits?");
  activeThermostat=new Berendsen();
}

ThermostatTraits<Thermostat> *ThermoStatContainer::getTraitByName(const std::string name){
  if(!availThermostats.count(name))
    return 0;
  return availThermostats[name];
}

Thermostat *ThermoStatContainer::makeByName(const std::string name,class ConfigFileBuffer * const fb){
  ThermostatTraits<Thermostat>* trait = getTraitByName(name);
  if(trait){
    Thermostat *res = trait->make(fb);
    res->addToContainer(this);
    return res;
  }
  else{
    return 0;
  }
}

void ThermoStatContainer::makeActive(const std::string name,class ConfigFileBuffer * const fb){
  Thermostat* newThermostat = makeByName(name,fb);
  if(newThermostat){
    if(activeThermostat){
      delete activeThermostat;
    }
    activeThermostat = newThermostat;
  }
}

void ThermoStatContainer::chooseNone(){
  if(activeThermostat){
    delete activeThermostat;
  }
  activeThermostat = new NoThermostat();
}

/** Destructor for Class ThermoStatContainer.
 *
 */
ThermoStatContainer::~ThermoStatContainer()
{
  if(activeThermostat){
    delete activeThermostat;
  }
  for(traitsMap::iterator iter= availThermostats.begin();iter!=availThermostats.end();++iter){
    delete (iter->second);
  }
}


