/*
 * BoostGraphCreator.hpp
 *
 *  Created on: May 17, 2017
 *      Author: heber
 */


#ifndef GRAPH_BOOSTGRAPHCREATOR_HPP_
#define GRAPH_BOOSTGRAPHCREATOR_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <map>
#include <vector>

#include <boost/function.hpp>
#include <boost/graph/adjacency_list.hpp>

#include "types.hpp"

class atom;
class bond;
class molecule;

class BoostGraphCreatorTest;
class BreadthFirstSearchGathererTest;

/** This is a helper class that contains functions to create a boost::graph
 * from the present bond graph of molecules.
 */
struct BoostGraphCreator
{
  //!> grant unit test access to private parts
  friend class BoostGraphCreatorTest;
  //!> grant unit test access to private parts that use BoostGraphCreator's internal graph
  friend class BreadthFirstSearchGathererTest;

public:

  //!> typedef for an undirected graph using boost::graph
  typedef boost::adjacency_list < boost::vecS, boost::vecS, boost::undirectedS,
      boost::property<boost::vertex_name_t, atomId_t>,
      boost::property<boost::vertex_color_t, boost::default_color_type> /* needed for limited-depth DFS,
      otherwise the property_map gets full size of graph */
  > UndirectedGraph;

  //!> typedef for a map of graph node indices
  typedef boost::property_map < UndirectedGraph, boost::vertex_index_t >::type index_map_t;
  typedef boost::property_map < UndirectedGraph, boost::vertex_index_t >::const_type const_index_map_t;
  //!> typedef for a map of graph node indices
  typedef boost::property_map < UndirectedGraph, boost::vertex_name_t >::type name_map_t;
  typedef boost::property_map < UndirectedGraph, boost::vertex_name_t >::const_type const_name_map_t;
  //!> typedef for the  predicate to evaluate for adding the current edge or not
  typedef boost::function<bool (const bond &)> predicate_t;
  //!> typedef for a Vertex
  typedef boost::graph_traits<UndirectedGraph>::vertex_descriptor Vertex;
  //!> typedef for vertex iterator
  typedef boost::graph_traits<UndirectedGraph>::vertex_iterator vertex_iter;
  //!> typedef for a Edge
  typedef boost::graph_traits<UndirectedGraph>::edge_descriptor Edge;
  //!> typedef for edge iterator
  typedef boost::graph_traits<UndirectedGraph>::edge_iterator edge_iter;

  //!> typedef for a node id
  typedef size_t nodeId_t;
  //!> typedef for map converting between node id in graph and the associated atomic id
  typedef std::map<atomId_t, nodeId_t> atomids_nodeids_t;

  /** Creates the boost::graph using all atoms and bonds in the given \a _mol.
   *
   * \param _mol molecule whose bond graph to construct
   * \param _pred predicate to evaluate on adding each edge/bond
   */
  void createFromMolecule(
      const molecule &_mol,
      const predicate_t &_pred);

  /** Creates the boost::graph using all atoms and bonds in the given vector
   * of \a _atoms
   *
   * \param _atoms vector of _atoms whose bond graph to construct
   * \param _pred predicate to evaluate on adding each edge/bond
   */
  void createFromAtoms(
      const std::vector<atom *> &_atoms,
      const predicate_t &_pred);

  /** Getter for the created graph.
   *
   * \return graph
   */
  UndirectedGraph get() const
  { return graph; }

  /** Getter for the index map  of the created graph.
   *
   * \return indexmap
   */
  index_map_t getIndexMap() const {
    return boost::get(boost::vertex_index, graph);
  }

  /** Return the number of vertices contained in the created graph.
   *
   * \return number of vertices
   */
  size_t getNumVertices() const {
    return boost::num_vertices(graph);
  }

  /** Return the number of edges contained in the created graph.
   *
   * \return number of edges
   */
  size_t getNumEdges() const {
    return boost::num_edges(graph);
  }

  /** Returns the node id to a given atom id \a _atomid.
   *
   * \param _atomid atom id
   * \return node id
   */
  nodeId_t getNodeId(const atomId_t &_atomid) const;

  /** General purpose function that contains the internal logic of walking the
   * bonds of a set of atoms given by \a _begin and \a _end iterators and
   * adding its edges to a graph based on the evaluation of a given predicate
   * \a _pred.
   *
   * \note We need \a _no_nodes because molecule::iterator does not work with
   * std::distance.
   *
   * \param _begin begin iterator
   * \param _end end iterator
   * \param _no_nodes number of nodes
   * \param _pred predicate
   */
  template <typename iterator>
  void createFromRange(
      const iterator &_begin,
      const iterator &_end,
      const size_t &_no_nodes,
      const predicate_t &_pred
      );

  /** Finds a given edge by its two atomic indices.
   *
   * \param _firstid first atomic id of edge
   * \param _secondid second atomic id of edge
   * \return edge descriptor in graph or empty descriptor
   */
  Edge findEdge(const atomId_t &_firstid, const atomId_t &_secondid);

  /** Allows to remove a present edge in the graph.
   *
   * \param _firstid first atomic id of edge
   * \param _secondid second atomic id of edge
   * \return true - edge found and removed, false - else
   */
  bool removeEdge(const atomId_t &_firstid, const atomId_t &_secondid);

  /** Adds an edge to the graph if not already present.
   *
   * \param _firstid first atomic id of edge
   * \param _secondid second atomic id of edge
   * \return true - edge not found thus added, false - else
   */
  bool addEdge(const atomId_t &_firstid, const atomId_t &_secondid);

private:
  //!> internal graph that is created by creator functions
  UndirectedGraph graph;
  //!> external property map for all the atomic ids of each graph node
  atomids_nodeids_t atomids_nodeids;
};

#include "BoostGraphCreator_impl.hpp"


#endif /* GRAPH_BOOSTGRAPHCREATOR_HPP_ */
