/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * MatrixContent.cpp
 *
 *  Created on: Nov 14, 2010
 *      Author: heber
 */


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "CodePatterns/Assert.hpp"
#include "defs.hpp"
#include "fast_functions.hpp"
#include "MatrixContent.hpp"
#include "RealSpaceMatrix.hpp"
#include "Vector.hpp"
#include "VectorContent.hpp"

#include <gsl/gsl_blas.h>
#include <gsl/gsl_eigen.h>
#include <gsl/gsl_linalg.h>
#include <gsl/gsl_matrix.h>
#include <gsl/gsl_multimin.h>
#include <gsl/gsl_vector.h>
#include <cmath>
#include <iostream>
#include <limits>
#include <set>


/** Constructor for class MatrixContent.
 *
 * This cstor is only present for serialization purposes.
 *
 */
MatrixContent::MatrixContent() :
    MatrixDimension(0,0),
    content(NULL),
    free_content_on_exit(true)
{
}

/** Constructor for class MatrixContent.
 * \param rows number of rows
 * \param columns number of columns
 */
MatrixContent::MatrixContent(size_t _rows, size_t _columns) :
    MatrixDimension(_rows,_columns),
    free_content_on_exit(true)
{
  content = gsl_matrix_calloc(getRows(), getColumns());
}

/** Constructor of class VectorContent.
 * We need this MatrixBaseCase for the VectorContentView class.
 * There no content should be allocated, as it is just a view with an internal
 * gsl_vector_view. Hence, MatrixBaseCase is just dummy class to give the
 * constructor a unique signature.
 * \param MatrixBaseCase
 */
MatrixContent::MatrixContent(size_t _rows, size_t _columns, MatrixBaseCase) :
    MatrixDimension(_rows,_columns),
    free_content_on_exit(true)
{}

/** Constructor for class MatrixContent.
 * \param rows number of rows
 * \param columns number of columns
 * \param *src array with components to initialize matrix with
 */
MatrixContent::MatrixContent(size_t _rows, size_t _columns, const double *src) :
    MatrixDimension(_rows,_columns),
    free_content_on_exit(true)
{
  content = gsl_matrix_calloc(getRows(), getColumns());
  set(0,0, src[0]);
  set(1,0, src[1]);
  set(2,0, src[2]);

  set(0,1, src[3]);
  set(1,1, src[4]);
  set(2,1, src[5]);

  set(0,2, src[6]);
  set(1,2, src[7]);
  set(2,2, src[8]);
}

/** Constructor that parses matrix from a stream.
 *
 * \note Matrix dimensions can be preparsed via
 * MatrixContent::preparseMatrixDimensions() without harming the stream.
 *
 * \param _row number of rows
 * \param _column number of columns
 * \param &inputstream stream to parse from
 */
MatrixContent::MatrixContent(size_t _row, size_t _column, std::istream &inputstream) :
  MatrixDimension(_row,_column),
  content(NULL),
  free_content_on_exit(true)
{
   // allocate matrix and set contents
  content = gsl_matrix_alloc(getRows(), getColumns());

  size_t row = 0;
  do {
    std::string line;
    getline(inputstream, line);
    //std::cout << line << std::endl;
    std::stringstream parseline(line);
    // skip comments
    if ((parseline.peek() == '#') || (parseline.str().empty()))
      continue;
    // break on empty lines
    if (parseline.peek() == '\n')
      break;

    // parse line with values
    std::vector<double> line_of_values;
    do {
      double value;
      parseline >> value >> ws;
      line_of_values.push_back(value);
    } while (parseline.good());

    // check number of columns parsed
    ASSERT(line_of_values.size() == getColumns(),
        "MatrixContent::MatrixContent() - row "
        +toString(row)+" has a different number of columns "
        +toString(line_of_values.size())+" than others before "
        +toString(getColumns())+".");

    for (size_t column = 0; column < getColumns(); ++column)
      set(row, column, line_of_values[column]);
    ++row;
  } while (inputstream.good());
  // check number of rows parsed
  ASSERT(row == getRows(),
      "MatrixContent::MatrixContent() - insufficent number of rows "
      +toString(row)+" < "+toString(getRows())+" read.");
}


/** Constructor for class MatrixContent.
 *
 * \param *src source gsl_matrix vector to copy and put into this class
 */
MatrixContent::MatrixContent(gsl_matrix *&src) :
    MatrixDimension(src->size1,src->size2),
    free_content_on_exit(true)
{
  content = gsl_matrix_alloc(src->size1, src->size2);
  gsl_matrix_memcpy(content,src);
//  content = src;
//  src = NULL;
}

/** Copy constructor for class MatrixContent.
 * \param &src reference to source MatrixContent
 */
MatrixContent::MatrixContent(const MatrixContent &src) :
    MatrixDimension(src.getRows(),src.getColumns()),
    free_content_on_exit(true)
{
  content = gsl_matrix_alloc(src.getRows(), src.getColumns());
  gsl_matrix_memcpy(content,src.content);
}

/** Copy constructor for class MatrixContent.
 * \param *src pointer to source MatrixContent
 */
MatrixContent::MatrixContent(const MatrixContent *src) :
    MatrixDimension(src->getRows(),src->getColumns()),
    free_content_on_exit(true)
{
  ASSERT(src != NULL, "MatrixContent::MatrixContent - pointer to source matrix is NULL!");
  content = gsl_matrix_alloc(src->getRows(), src->getColumns());
  gsl_matrix_memcpy(content,src->content);
}

/** Destructor for class MatrixContent.
 */
MatrixContent::~MatrixContent()
{
  if (free_content_on_exit)
    gsl_matrix_free(content);
}

/** Return a VectorViewContent of the \a column -th column vector.
 *
 * @param column index of column
 * @return column of matrix as VectorContent
 */
VectorContent *MatrixContent::getColumnVector(size_t column) const
{
  ASSERT(column < getColumns(),
      "MatrixContent::getColumnVector() - requested column "+toString(column)
      +" greater than dimension "+toString(getColumns()));
  return (new VectorViewContent(gsl_matrix_column(content,column)));
}

/** Returns a VectorViewContent of the \a row -th row vector.
 * @param row row index
 * @return VectorContent of row vector
 */
VectorContent *MatrixContent::getRowVector(size_t row) const
{
  ASSERT(row < getRows(),
      "MatrixContent::getColumnVector() - requested row "+toString(row)
      +" greater than dimension "+toString(getRows()));
  return (new VectorViewContent(gsl_matrix_row(content,row)));
}

/** Returns the main diagonal of the matrix as VectorContent.
 * @return diagonal as VectorContent.
 */
VectorContent *MatrixContent::getDiagonalVector() const
{
  return (new VectorViewContent(gsl_matrix_diagonal(content)));
}

/** Set matrix to identity.
 */
void MatrixContent::setIdentity()
{
  for(int i=getRows();i--;){
    for(int j=getColumns();j--;){
      set(i,j,(double)(i==j));
    }
  }
}

/** Set all matrix components to zero.
 */
void MatrixContent::setZero()
{
  for(int i=getRows();i--;){
    for(int j=getColumns();j--;){
      set(i,j,0.);
    }
  }
}

/** Set all matrix components to a given value.
 * \param _value value to set each component to
 */
void MatrixContent::setValue(double _value)
{
  for(int i=getRows();i--;){
    for(int j=getColumns();j--;){
      set(i,j,_value);
    }
  }
}

/** Copy operator for MatrixContent with self-assignment check.
 * \param &src matrix to compare to
 * \return reference to this
 */
MatrixContent &MatrixContent::operator=(const MatrixContent &src)
{
  if(&src!=this){
    gsl_matrix_memcpy(content,src.content);
  }
  return *this;
}

/** Addition operator.
 * \param &rhs matrix to add
 * \return reference to this
 */
const MatrixContent &MatrixContent::operator+=(const MatrixContent &rhs)
{
  gsl_matrix_add(content, rhs.content);
  return *this;
}

/** Subtraction operator.
 * \param &rhs matrix to subtract
 * \return reference to this
 */
const MatrixContent &MatrixContent::operator-=(const MatrixContent &rhs)
    {
  gsl_matrix_sub(content, rhs.content);
  return *this;
}

/** Multiplication operator.
 * Note that here matrix have to have same dimensions.
 * \param &rhs matrix to multiply with
 * \return reference to this
 */
const MatrixContent &MatrixContent::operator*=(const MatrixContent &rhs)
{
  ASSERT(rhs.getColumns() == rhs.getRows(),
      "MatrixContent::operator*=() - rhs matrix is not square: "+toString(rhs.getColumns())+" != "+toString(rhs.getRows())+".");
  ASSERT(getColumns() == rhs.getRows(),
      "MatrixContent::operator*=() - columns dimension differ: "+toString(getColumns())+" != "+toString(rhs.getRows())+".");
  (*this) = (*this)*rhs;
  return *this;
}

/** Multiplication with copy operator.
 * \param &rhs matrix to multiply with
 * \return reference to newly allocated MatrixContent
 */
const MatrixContent MatrixContent::operator*(const MatrixContent &rhs) const
{
  ASSERT (getColumns() == rhs.getRows(),
      "MatrixContent::operator*() - dimensions not match for matrix product (a,b)*(b,c) = (a,c):"
      "("+toString(getRows())+","+toString(getColumns())+")*("+toString(rhs.getRows())+","+toString(rhs.getColumns())+")");
  gsl_matrix *res = gsl_matrix_alloc(getRows(), rhs.getColumns());
  gsl_blas_dgemm(CblasNoTrans, CblasNoTrans, 1.0, content, rhs.content, 0.0, res);
  // gsl_matrix is taken over by constructor, hence no free
  MatrixContent tmp(res);
  gsl_matrix_free(res);
  return tmp;
}

/** Hadamard multiplication with copy operator.
 * The Hadamard product is component-wise matrix product.
 * \param &rhs matrix to hadamard-multiply with
 * \return reference to newly allocated MatrixContent
 */
const MatrixContent MatrixContent::operator&(const MatrixContent &rhs) const
{
  ASSERT ((getRows() == rhs.getRows()) && (getColumns() == rhs.getColumns()),
      "MatrixContent::operator&() - dimensions not match for matrix product (a,b) != (b,c):"
      "("+toString(getRows())+","+toString(getColumns())+") != ("+toString(rhs.getRows())+","+toString(rhs.getColumns())+")");
  gsl_matrix *res = gsl_matrix_alloc(getRows(), rhs.getColumns());
  for (size_t i=0;i<getRows();++i)
    for (size_t j=0;j<getColumns();++j)
      gsl_matrix_set(res, i,j, gsl_matrix_get(content, i,j)*gsl_matrix_get(rhs.content, i,j));
  // gsl_matrix is taken over by constructor, hence no free
  MatrixContent tmp(res);
  gsl_matrix_free(res);
  return tmp;
}

/** Hadamard multiplication with copy operator.
 * The Hadamard product is component-wise matrix product.
 * Note that Hadamard product can easily be done on top of \a *this matrix.
 * Hence, we don't need to use the multiply and copy operator as in the case of
 * MatrixContent::operator*=().
 * \param &rhs matrix to hadamard-multiply with
 * \return reference to newly allocated MatrixContent
 */
const MatrixContent &MatrixContent::operator&=(const MatrixContent &rhs)
{
  ASSERT ((getRows() == rhs.getRows()) && (getColumns() == rhs.getColumns()),
      "MatrixContent::operator&() - dimensions not match for matrix product (a,b) != (b,c):"
      "("+toString(getRows())+","+toString(getColumns())+") != ("+toString(rhs.getRows())+","+toString(rhs.getColumns())+")");
  for (size_t i=0;i<getRows();++i)
    for (size_t j=0;j<getColumns();++j)
      gsl_matrix_set(content, i,j, gsl_matrix_get(content, i,j)*gsl_matrix_get(rhs.content, i,j));
  return *this;
}

/* ========================== Accessing =============================== */

/** Accessor for manipulating component (i,j).
 * \param i row number
 * \param j column number
 * \return reference to component (i,j)
 */
double &MatrixContent::at(size_t i, size_t j)
{
  ASSERT((i>=0) && (i<getRows()),
      "MatrixContent::at() - Index i="+toString(i)+" for Matrix access out of range [0,"+toString(getRows())+"]");
  ASSERT((j>=0) && (j<getColumns()),
      "MatrixContent::at() - Index j="+toString(j)+" for Matrix access out of range [0,"+toString(getColumns())+"]");
  return *gsl_matrix_ptr (content, i, j);
}

/** Constant accessor for (value of) component (i,j).
 * \param i row number
 * \param j column number
 * \return const component (i,j)
 */
const double MatrixContent::at(size_t i, size_t j) const
{
  ASSERT((i>=0) && (i<getRows()),
      "MatrixContent::at() - Index i="+toString(i)+" for Matrix access out of range [0,"+toString(getRows())+"]");
  ASSERT((j>=0) && (j<getColumns()),
      "MatrixContent::at() - Index j="+toString(j)+" for Matrix access out of range [0,"+toString(getColumns())+"]");
  return gsl_matrix_get(content, i, j);
}

/** These functions return a pointer to the \a m-th element of a matrix.
 *  If \a m or \a n lies outside the allowed range of 0 to MatrixContent::dimension-1 then the error handler is invoked and a null pointer is returned.
 * \param m index
 * \return pointer to \a m-th element
 */
double *MatrixContent::Pointer(size_t m, size_t n)
{
  return gsl_matrix_ptr (content, m, n);
};

/** These functions return a constant pointer to the \a m-th element of a matrix.
 *  If \a m or \a n lies outside the allowed range of 0 to MatrixContent::dimension-1 then the error handler is invoked and a null pointer is returned.
 * \param m index
 * \return const pointer to \a m-th element
 */
const double *MatrixContent::const_Pointer(size_t m, size_t n) const
{
  return gsl_matrix_const_ptr (content, m, n);
};

/* ========================== Initializing =============================== */

/** Setter for component (i,j).
 * \param i row numbr
 * \param j column numnber
 * \param value value to set componnt (i,j) to
 */
void MatrixContent::set(size_t i, size_t j, const double value)
{
  ASSERT((i>=0) && (i<getRows()),
      "MatrixContent::set() - Index i="+toString(i)+" for Matrix access out of range [0,"+toString(getRows())+"]");
  ASSERT((j>=0) && (j<getColumns()),
      "MatrixContent::set() - Index j="+toString(j)+" for Matrix access out of range [0,"+toString(getColumns())+"]");
  gsl_matrix_set(content,i,j,value);
}

/** This function sets the matrix from a double array.
 * Creates a matrix view of the array and performs a memcopy.
 * \param *x array of values (no dimension check is performed)
 */
void MatrixContent::setFromDoubleArray(double * x)
{
  gsl_matrix_view m = gsl_matrix_view_array (x, getRows(), getColumns());
  gsl_matrix_memcpy (content, &m.matrix);
};

/* ====================== Exchanging elements ============================ */
/** This function exchanges the \a i-th and \a j-th row of the matrix in-place.
 * \param i i-th row to swap with ...
 * \param j ... j-th row to swap against
 */
bool MatrixContent::SwapRows(size_t i, size_t j)
{
  return (gsl_matrix_swap_rows (content, i, j) == GSL_SUCCESS);
};

/** This function exchanges the \a i-th and \a j-th column of the matrix in-place.
 * \param i i-th column to swap with ...
 * \param j ... j-th column to swap against
 */
bool MatrixContent::SwapColumns(size_t i, size_t j)
{
  return (gsl_matrix_swap_columns (content, i, j) == GSL_SUCCESS);
};

/** This function exchanges the \a i-th row and \a j-th column of the matrix in-place.
 * The matrix must be square for this operation to be possible.
 * \param i i-th row to swap with ...
 * \param j ... j-th column to swap against
 */
bool MatrixContent::SwapRowColumn(size_t i, size_t j)
{
  ASSERT (getRows() == getColumns(),
      "MatrixContent::SwapRowColumn() - The matrix must be square for swapping row against column to be possible.");
  return (gsl_matrix_swap_rowcol (content, i, j) == GSL_SUCCESS);
};

/** Return transposed matrix.
 * \return new matrix that is transposed of this.
 */
MatrixContent MatrixContent::transposed() const
{
  gsl_matrix *res = gsl_matrix_alloc(getColumns(), getRows());  // column and row dimensions exchanged!
  gsl_matrix_transpose_memcpy(res, content);
  MatrixContent newContent(res);
  gsl_matrix_free(res);
  return newContent;
}

/** Turn this matrix into its transposed.
 * Note that this is only possible if rows == columns.
 */
MatrixContent &MatrixContent::transpose()
{
  ASSERT( getRows() == getColumns(),
      "MatrixContent::transpose() - cannot transpose onto itself as matrix not square: "+toString(getRows())+"!="+toString(getColumns())+"!");
  double tmp;
  for (size_t i=0;i<getRows();i++)
    for (size_t j=i+1;j<getRows();j++) {
      tmp = at(j,i);
      at(j,i) = at(i,j);
      at(i,j) = tmp;
    }
  return *this;
}

/** Performs a SVD on the contained matrix.
 *
 * A SVD is the factorization of a MxN matrix A into the product form \f$U S V^t\f$, where
 *  -# U is a MxN orthogonal matrix
 *  -# S is a NxN diagonal matrix (here just a vector)
 *  -# V is a NxN orthogonal, square matrix.
 *
 * The SVD is only done if MatrixDimension::rows is larger equal to MatrixDimension::columns, i.e.
 * N >= M.
 *
 * \note The matrix content of this matrix, \a V and \a S are overwritten in the process.
 *
 * \param V orthogonal square matrix on return
 * \param S diagonal matrix on return
 */
void MatrixContent::performSingularValueDecomposition(MatrixContent &V, VectorContent &S)
{
  ASSERT(getRows() >= getColumns(),
      "MatrixContent::performSingularValueDecomposition() - row count "
      +toString(getRows())+" must be larger than or equal to column count "+toString(getColumns())+".");
  ASSERT(V.getRows() == V.getColumns(),
      "MatrixContent::performSingularValueDecomposition() - Matrix V must be square!");
  ASSERT(V.getRows() == getColumns(),
      "MatrixContent::performSingularValueDecomposition() - Given matrix V does not have the right dimension "
      +toString(V.getRows())+"!="+toString(getRows())+".");
  ASSERT(S.getDimension() == getColumns(),
      "MatrixContent::performSingularValueDecomposition() - Vector S does not have the right dimension "
      +toString(S.getDimension())+"!="+toString(getColumns())+".");
  gsl_matrix *A = content;
  gsl_vector *work = gsl_vector_alloc(getColumns());
  gsl_linalg_SV_decomp(A,V.content,S.content,work);
  gsl_vector_free(work);
}

/** Performs a SVD on the contained matrix and solves the linear system \f$Ax=b\f$ with given \a b.
 *
 * The SVD is only done if MatrixDimension::rows is larger equal to MatrixDimension::columns.
 *
 * \note The matrix content is overwritten in the process.
 *
 * \param V NxN orthogonal, diagonal matrix from SVD decomposition
 * \param S N vector representing the diagonal elements from SVD decomposition
 * \param *b right-hand side of linear equation \f$Ax=b\f$ to solve
 * \return *x solution, if rows>columns this will be the solution in the least squares sense \f$||Ax-b||^2\f$
 */
VectorContent MatrixContent::solveOverdeterminedLinearEquation(MatrixContent &V, VectorContent &S, const VectorContent &b)
{
  ASSERT(b.getDimension() == getRows(),
      "MatrixContent::performSingularValueDecomposition() - Given vector b does not have the right dimension "
      +toString(b.getDimension())+"!="+toString(getRows())+".");
  ASSERT(V.getRows() == V.getColumns(),
      "MatrixContent::performSingularValueDecomposition() - Matrix V must be square!");
  ASSERT(V.getRows() == getColumns(),
      "MatrixContent::performSingularValueDecomposition() - Given matrix V does not have the right dimension "
      +toString(V.getRows())+"!="+toString(getRows())+".");
  ASSERT(S.getDimension() == getColumns(),
      "MatrixContent::performSingularValueDecomposition() - Vector S does not haver the right dimension "
      +toString(S.getDimension())+"!="+toString(getColumns())+".");

  gsl_vector *x = gsl_vector_alloc(getColumns());
  gsl_linalg_SV_solve(content,V.content,S.content,b.content,x);
  return VectorContent(x, true);
}

/** Performs a SVD on the contained matrix and solves the linear system \f$Ax=b\f$ with given \a b.
 *
 * The SVD is only done if MatrixDimension::rows is larger equal to MatrixDimension::columns.
 *
 * \note The matrix content is overwritten in the process.
 *
 * \param *b right-hand side of linear equation \f$Ax=b\f$ to solve.
 * \return *x solution, if rows>columns this will be the solution in the least squares sense \f$||Ax-b||^2\f$
 */
VectorContent MatrixContent::solveOverdeterminedLinearEquation(const VectorContent &b)
{
  ASSERT(b.getDimension() == getRows(),
      "MatrixContent::performSingularValueDecomposition() - Given vector b does not have the right dimension "
      +toString(b.getDimension())+"!="+toString(getRows())+".");
  MatrixContent V(getColumns(), getColumns());
  VectorContent S(getColumns());
  performSingularValueDecomposition(V,S);
  return solveOverdeterminedLinearEquation(V,S,b);
}

/** Transform the matrix to its eigenbasis and return resulting eigenvalues.
 * Note that we only return real-space part in case of non-symmetric matrix.
 * \warn return vector has to be freed'd
 * TODO: encapsulate return value in boost::shared_ptr or in VectorContent.
 * \return gsl_vector pointer to vector of eigenvalues
 */
gsl_vector* MatrixContent::transformToEigenbasis()
{
  if (getRows() == getColumns()) { // symmetric
    gsl_eigen_symmv_workspace *T = gsl_eigen_symmv_alloc(getRows());
    gsl_vector *eval = gsl_vector_alloc(getRows());
    gsl_matrix *evec = gsl_matrix_alloc(getRows(), getRows());
    gsl_eigen_symmv(content, eval, evec, T);
    gsl_eigen_symmv_free(T);
    gsl_matrix_memcpy(content, evec);
    gsl_matrix_free(evec);
    return eval;
  } else { // non-symmetric
    // blow up gsl_matrix in content to square matrix, fill other components with zero
    const size_t greaterDimension = getRows() > getColumns() ? getRows() : getColumns();
    gsl_matrix *content_square = gsl_matrix_alloc(greaterDimension, greaterDimension);
    for (size_t i=0; i<greaterDimension; i++) {
      for (size_t j=0; j<greaterDimension; j++) {
        const double value = ((i < getRows()) && (j < getColumns())) ? gsl_matrix_get(content,i,j) : 0.;
        gsl_matrix_set(content_square, i,j, value);
      }
    }

    // show squared matrix by putting it into a MatrixViewContent
    MatrixContent *ContentSquare = new MatrixViewContent(gsl_matrix_submatrix(content_square,0,0,content_square->size1, content_square->size2));
    std::cout << "The squared matrix is " << *ContentSquare << std::endl;

    // solve eigenvalue problem
    gsl_eigen_nonsymmv_workspace *T = gsl_eigen_nonsymmv_alloc(getRows());
    gsl_vector_complex *eval = gsl_vector_complex_alloc(greaterDimension);
    gsl_matrix_complex *evec = gsl_matrix_complex_alloc(greaterDimension, greaterDimension);
    gsl_eigen_nonsymmv(content_square, eval, evec, T);
    gsl_eigen_nonsymmv_free(T);

    // copy eigenvectors real-parts into content_square and ...
    for (size_t i=0; i<greaterDimension; i++)
      for (size_t j=0; j<greaterDimension; j++)
        gsl_matrix_set(content_square, i,j, GSL_REAL(gsl_matrix_complex_get(evec,i,j)));

    // ... show complex-valued eigenvector matrix
    std::cout << "The real-value eigenvector matrix is " << *ContentSquare << std::endl;
//    std::cout << "Resulting eigenvector matrix is [";
//    for (size_t i=0; i<greaterDimension; i++) {
//      for (size_t j=0; j<greaterDimension; j++) {
//        std::cout << "(" << GSL_REAL(gsl_matrix_complex_get(evec,i,j))
//            << "," << GSL_IMAG(gsl_matrix_complex_get(evec,i,j)) << ")";
//        if (j < greaterDimension-1)
//          std::cout << " ";
//      }
//      if (i < greaterDimension-1)
//        std::cout << "; ";
//    }
//    std::cout << "]" << std::endl;

    // copy real-parts of complex eigenvalues and eigenvectors (column-wise orientation)
    gsl_vector *eval_real = gsl_vector_alloc(getColumns());
    size_t I=0;
    for (size_t i=0; i<greaterDimension; i++) { // only copy real space part
      if (fabs(GSL_REAL(gsl_vector_complex_get(eval,i))) > LINALG_MYEPSILON()) { // only take eigenvectors with value > 0
        std::cout << i << "th eigenvalue is (" << GSL_REAL(gsl_vector_complex_get(eval,i)) << "," << GSL_IMAG(gsl_vector_complex_get(eval,i)) << ")" << std::endl;
        for (size_t j=0; j<greaterDimension; j++) {
          if (fabs(GSL_IMAG(gsl_matrix_complex_get(evec,j,i))) > LINALG_MYEPSILON())
            std::cerr << "MatrixContent::transformToEigenbasis() - WARNING: eigenvectors are complex-valued!" << std::endl;
          gsl_matrix_set(content, j,I, GSL_REAL(gsl_matrix_complex_get(evec,j,i)));
        }
        if (fabs(GSL_IMAG(gsl_vector_complex_get(eval,I))) > LINALG_MYEPSILON())
          std::cerr << "MatrixContent::transformToEigenbasis() - WARNING: eigenvectors are complex-valued!" << std::endl;
        gsl_vector_set(eval_real, I, GSL_REAL(gsl_vector_complex_get(eval, i)));
        I++;
      }
    }
    gsl_matrix_complex_free(evec);
    gsl_vector_complex_free(eval);
    delete ContentSquare;

    return eval_real;
  }
}


/** Sorts the eigenpairs in ascending order of the eigenvalues.
 * We assume that MatrixContent::transformToEigenbasis() has just been called.
 * @param eigenvalues vector of eigenvalue from
 *        MatrixContent::transformToEigenbasis()
 */
void MatrixContent::sortEigenbasis(gsl_vector *eigenvalues)
{
  gsl_eigen_symmv_sort (eigenvalues, content,
                          GSL_EIGEN_SORT_VAL_ASC);
}

/* ============================ Properties ============================== */
/** Checks whether matrix' elements are strictly null.
 * \return true - is null, false - else
 */
bool MatrixContent::IsNull() const
{
  return gsl_matrix_isnull (content);
};

/** Checks whether matrix' elements are strictly positive.
 * \return true - is positive, false - else
 */
bool MatrixContent::IsPositive() const
{
  return gsl_matrix_ispos (content);
};

/** Checks whether matrix' elements are strictly negative.
 * \return true - is negative, false - else
 */
bool MatrixContent::IsNegative() const
{
  return gsl_matrix_isneg (content);
};

/** Checks whether matrix' elements are strictly non-negative.
 * \return true - is non-negative, false - else
 */
bool MatrixContent::IsNonNegative() const
{
  return gsl_matrix_isnonneg (content);
};

/** This function performs a Cholesky decomposition to determine whether matrix is positive definite.
 * We check whether GSL returns GSL_EDOM as error, indicating that decomposition failed due to matrix not being positive-definite.
 * \return true - matrix is positive-definite, false - else
 */
bool MatrixContent::IsPositiveDefinite() const
{
  if (getRows() != getColumns())  // only possible for square matrices.
    return false;
  else
    return (gsl_linalg_cholesky_decomp (content) != GSL_EDOM);
};


/** Calculates the determinant of the matrix.
 * if matrix is square, uses LU decomposition.
 */
double MatrixContent::Determinant() const
{
  int signum = 0;
  ASSERT(getRows() == getColumns(),
      "MatrixContent::Determinant() - determinant can only be calculated for square matrices.");
  gsl_permutation *p = gsl_permutation_alloc(getRows());
  gsl_linalg_LU_decomp(content, p, &signum);
  gsl_permutation_free(p);
  return gsl_linalg_LU_det(content, signum);
};

/** Preparses a matrix in an input stream for row and column count.
 *
 *  \note This does not change the get position within the stream.
 *
 * @param inputstream to parse matrix from
 * @return pair with (row, column)
 */
std::pair<size_t, size_t> MatrixContent::preparseMatrixDimensions(std::istream &inputstream)
{
  const std::streampos initial_position(inputstream.tellg());
//  std::cout << "We are at position "
//      +toString((int)inputstream.tellg())+" from start of stream." << std::endl;
  size_t rows = 0;
  size_t columns = 0;
  do {
    std::string line;
    getline(inputstream, line);
    std::stringstream parseline(line);
    // skip comments
    if ((parseline.peek() == '#') || (line.empty()))
      continue;
    // break on empty lines
    if (parseline.peek() == '\n')
      break;

    // parse line with values
    std::vector<double> line_of_values;
    do {
      double value;
      parseline >> value >> ws;
      line_of_values.push_back(value);
    } while (parseline.good());

    // set matrixdimension if first line
    if (columns == 0) {
      columns = line_of_values.size();
    } else { // otherwise check
      ASSERT(columns == line_of_values.size(),
          "MatrixContent::MatrixContent() - row "
          +toString(rows)+" has a different number of columns "
          +toString(line_of_values.size())+" than this matrix has "
          +toString(columns)+".");
    }
    ++rows;
  } while (inputstream.good());
  // clear end of file flags
  inputstream.clear();

  // reset to initial position
  inputstream.seekg(initial_position);
//  std::cout << "We are again at position "
//      +toString((int)inputstream.tellg())+" from start of stream." << std::endl;

  return make_pair(rows, columns);
}

/** Writes matrix content to ostream in such a manner that it can be re-parsed
 * by the code in the constructor.
 *
 * @param ost stream to write to
 */
void MatrixContent::write(std::ostream &ost) const
{
  for (size_t row = 0; row < getRows(); ++row) {
    for (size_t column = 0; column < getColumns(); ++column)
      ost << at(row, column) << "\t";
    ost << std::endl;
  }
}


/* ============================= Operators =============================== */

/** Scalar multiplication operator.
 * \param factor factor to scale with
 */
const MatrixContent &MatrixContent::operator*=(const double factor)
{
  gsl_matrix_scale(content, factor);
  return *this;
}

/** Scalar multiplication and copy operator.
 * \param factor factor to scale with
 * \param &mat MatrixContent to scale
 * \return copied and scaled MatrixContent
 */
const MatrixContent operator*(const double factor,const MatrixContent& mat)
{
  MatrixContent tmp = mat;
  tmp*=factor;
  return tmp;
}

/** Scalar multiplication and copy operator (with operands exchanged).
 * \param &mat MatrixContent to scale
 * \param factor factor to scale with
 * \return copied and scaled MatrixContent
 */
const MatrixContent operator*(const MatrixContent &mat,const double factor)
{
  return factor*mat;
}

/** Sums two MatrixContents \a  and \b component-wise.
 * \param a first MatrixContent
 * \param b second MatrixContent
 * \return a + b
 */
MatrixContent const operator+(const MatrixContent& a, const MatrixContent& b)
{
  ASSERT(a.getRows() == b.getRows(), "MatrixContent::operator+() - row counts have to match: "
      +toString(a.getRows())+" != "+toString(b.getRows())+"!");
  ASSERT(a.getColumns() == b.getColumns(), "MatrixContent::operator+() - column counts have to match: "
      +toString(a.getColumns())+" != "+toString(b.getColumns())+"!");
  MatrixContent x(a);
  x += b;
  return x;
};

/** Subtracts MatrixContent \a from \b component-wise.
 * \param a first MatrixContent
 * \param b second MatrixContent
 * \return a - b
 */
MatrixContent const operator-(const MatrixContent& a, const MatrixContent& b)
{
  ASSERT(a.getRows() == b.getRows(), "MatrixContent::operator+() - row counts have to match: "
      +toString(a.getRows())+" != "+toString(b.getRows())+"!");
  ASSERT(a.getColumns() == b.getColumns(), "MatrixContent::operator+() - column counts have to match: "
      +toString(a.getColumns())+" != "+toString(b.getColumns())+"!");
  MatrixContent x(a);
  x -= b;
  return x;
};

/** Equality operator.
 * Note that we use numerical sensible checking, i.e. with threshold LINALG_MYEPSILON().
 * \param &rhs MatrixContent to checks against
 */
bool MatrixContent::operator==(const MatrixContent &rhs) const
    {
  if ((getRows() == rhs.getRows()) && (getColumns() == rhs.getColumns())) {
    for(int i=getRows();i--;){
      for(int j=getColumns();j--;){
        if(fabs(at(i,j)-rhs.at(i,j))>LINALG_MYEPSILON()){
          return false;
        }
      }
    }
    return true;
  }
  return false;
}


std::ostream & operator<<(std::ostream &ost, const MatrixContent &mat)
{
  ost << "\\begin{pmatrix}";
  for (size_t i=0;i<mat.getRows();i++) {
    for (size_t j=0;j<mat.getColumns();j++) {
      ost << mat.at(i,j) << " ";
      if (j != mat.getColumns()-1)
        ost << "& ";
    }
    if (i != mat.getRows()-1)
      ost << "\\\\ ";
  }
  ost << "\\end{pmatrix}";
  return ost;
}
