/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * LineSegment.cpp
 *
 *  Created on: Jul 22, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "LineSegment.hpp"
#include "CodePatterns/Assert.hpp"

#include "Line.hpp"
#include "Vector.hpp"

using namespace std;

LineSegment::LineSegment(const Vector &point1,const Vector &point2){
  line.reset(new Line(point1,point1-point2));
  begin.reset(new LinePoint(line->getLinePoint(point1)));
  end.reset(new LinePoint(line->getLinePoint(point2)));
}

LineSegment::LineSegment(const LinePoint &lp1,const LinePoint &lp2){
  ASSERT(lp1.getLine()==lp2.getLine(),"Cannot make a lineSegment with two different lines");
  line.reset(new Line(lp1.getLine()));
  begin.reset(new LinePoint(lp1));
  end.reset(new LinePoint(lp2));
}
LineSegment::LineSegment(const LineSegment &src)
{
  line.reset(new Line(*src.line));
  begin.reset(new LinePoint(*src.begin));
  end.reset(new LinePoint(*src.end));
}
LineSegment::~LineSegment(){
}
LineSegment &LineSegment::operator=(const LineSegment &src){
  if(this!=&src){
    lineptr _line(new Line(*src.line));
    pointptr _begin(new LinePoint(*src.begin));
    pointptr _end(new LinePoint(*src.end));
    line = _line;
    begin = _begin;
    end = _end;
  }
  return *this;
}

bool LineSegment::isContained(const Vector &point) const{
  if(!line->isContained(point)){
    return false;
  }
  LinePoint lp = line->getLinePoint(point);
  return *begin <= lp && lp <= *end;
}

bool LineSegment::isContained(const LinePoint &lp) const{
  ASSERT(lp.getLine()==*line,"Cannot compare Linepoints from different Lines");
  return *begin <= lp && lp <= *end;
}

bool LineSegment::overlaps(const LineSegment &lineSeg) const{
  ASSERT(*lineSeg.line == *line,"Cannot compare Linesegments from different Lines");
  // mutual comparison of endpoints
  return lineSeg.isContained(*end) || isContained(*lineSeg.end);
}

bool LineSegment::hasZeroLength() const{
  return *begin==*end;
}

LinePoint LineSegment::getBegin() const{
  return *begin;
}

LinePoint LineSegment::getEnd() const{
  return *end;
}

Line LineSegment::getLine() const{
  return *line;
}

bool operator==(const LineSegment &x,const LineSegment &y){
  ASSERT(*x.line==*y.line,"Comparison of Linesegments from different lines");
  return *x.begin == *y.begin && *x.end == *y.end;
}
bool operator<(const LineSegment &x,const LineSegment &y){
  ASSERT(*x.line==*y.line,"Comparison of Linesegments from different lines");
  return (*x.begin != *y.begin) ? *x.begin<*y.begin:*x.end<*y.end;
}
