/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * Please see the COPYING file or "Copyright notice" in builder.cpp for details.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>. 
 */

/*
 * PairPotential_Harmonic.cpp
 *
 *  Created on: Sep 26, 2012
 *      Author: heber
 */


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "PairPotential_Harmonic.hpp"

#include <boost/assign/list_of.hpp> // for 'map_list_of()'
#include <boost/bind.hpp>
#include <boost/lambda/lambda.hpp>
#include <string>

#include "CodePatterns/Assert.hpp"

#include "FunctionApproximation/Extractors.hpp"
#include "FunctionApproximation/TrainingData.hpp"
#include "Potentials/helpers.hpp"
#include "Potentials/ParticleTypeCheckers.hpp"

class Fragment;

// static definitions
const PairPotential_Harmonic::ParameterNames_t
PairPotential_Harmonic::ParameterNames =
      boost::assign::list_of<std::string>
      ("spring_constant")
      ("equilibrium_distance")
    ;
const std::string PairPotential_Harmonic::potential_token("harmonic_bond");

PairPotential_Harmonic::PairPotential_Harmonic() :
  EmpiricalPotential(),
  params(parameters_t(MAXPARAMS, 0.))
{
  // have some decent defaults for parameter_derivative checking
  params[spring_constant] = 1.;
  params[equilibrium_distance] = 1.;
}

PairPotential_Harmonic::PairPotential_Harmonic(
    const ParticleTypes_t &_ParticleTypes) :
  EmpiricalPotential(_ParticleTypes),
  params(parameters_t(MAXPARAMS, 0.))
{
  // have some decent defaults for parameter_derivative checking
  params[spring_constant] = 1.;
  params[equilibrium_distance] = 1.;
}

PairPotential_Harmonic::PairPotential_Harmonic(
    const ParticleTypes_t &_ParticleTypes,
    const double _spring_constant,
    const double _equilibrium_distance) :
  EmpiricalPotential(_ParticleTypes),
  params(parameters_t(MAXPARAMS, 0.))
{
  params[spring_constant] = _spring_constant;
  params[equilibrium_distance] = _equilibrium_distance;
}

void PairPotential_Harmonic::setParameters(const parameters_t &_params)
{
  const size_t paramsDim = _params.size();
  ASSERT( paramsDim <= getParameterDimension(),
      "PairPotential_Harmonic::setParameters() - we need not more than "
      +toString(getParameterDimension())+" parameters.");
  for(size_t i=0;i<paramsDim;++i)
    params[i] = _params[i];

#ifndef NDEBUG
  parameters_t check_params(getParameters());
  check_params.resize(paramsDim); // truncate to same size
  ASSERT( check_params == _params,
      "PairPotential_Harmonic::setParameters() - failed, mismatch in to be set "
      +toString(_params)+" and set "+toString(check_params)+" params.");
#endif
}

PairPotential_Harmonic::results_t
PairPotential_Harmonic::operator()(
    const arguments_t &arguments
    ) const
{
  ASSERT( arguments.size() == 1,
      "PairPotential_Harmonic::operator() - requires exactly one argument.");
  ASSERT( ParticleTypeChecker::checkArgumentsAgainstParticleTypes(
      arguments, getParticleTypes()),
      "PairPotential_Harmonic::operator() - types don't match with ones in arguments.");
  const argument_t &r_ij = arguments[0];
  const result_t result =
      params[spring_constant]
             * Helpers::pow( r_ij.distance - params[equilibrium_distance], 2 );
  return std::vector<result_t>(1, result);
}

PairPotential_Harmonic::derivative_components_t
PairPotential_Harmonic::derivative(
    const arguments_t &arguments
    ) const
{
  ASSERT( arguments.size() == 1,
      "PairPotential_Harmonic::operator() - requires exactly one argument.");
  ASSERT( ParticleTypeChecker::checkArgumentsAgainstParticleTypes(
      arguments, getParticleTypes()),
      "PairPotential_Harmonic::operator() - types don't match with ones in arguments.");
  derivative_components_t result;
  const argument_t &r_ij = arguments[0];
  result.push_back( 2. * params[spring_constant] * ( r_ij.distance - params[equilibrium_distance]) );
  ASSERT( result.size() == 1,
      "PairPotential_Harmonic::operator() - we did not create exactly one component.");
  return result;
}

PairPotential_Harmonic::results_t
PairPotential_Harmonic::parameter_derivative(
    const arguments_t &arguments,
    const size_t index
    ) const
{
  ASSERT( arguments.size() == 1,
      "PairPotential_Harmonic::parameter_derivative() - requires exactly one argument.");
  ASSERT( ParticleTypeChecker::checkArgumentsAgainstParticleTypes(
      arguments, getParticleTypes()),
      "PairPotential_Harmonic::operator() - types don't match with ones in arguments.");
  const argument_t &r_ij = arguments[0];
  switch (index) {
    case spring_constant:
    {
      const result_t result =
                 Helpers::pow( r_ij.distance - params[equilibrium_distance], 2 );
      return std::vector<result_t>(1, result);
      break;
    }
    case equilibrium_distance:
    {
      const result_t result =
          -2. * params[spring_constant]
                 * ( r_ij.distance - params[equilibrium_distance]);
      return std::vector<result_t>(1, result);
      break;
    }
    default:
      ASSERT(0, "PairPotential_Harmonic::parameter_derivative() - derivative to unknown parameter desired.");
      break;
  }

  return PairPotential_Harmonic::results_t(1, 0.);
}

FunctionModel::extractor_t
PairPotential_Harmonic::getFragmentSpecificExtractor() const
{
  Fragment::charges_t charges;
  charges.resize(getParticleTypes().size());
  std::transform(getParticleTypes().begin(), getParticleTypes().end(),
      charges.begin(), boost::lambda::_1);
  FunctionModel::extractor_t returnfunction =
      boost::bind(&Extractors::gatherDistancesFromFragment,
          boost::bind(&Fragment::getPositions, _1),
          boost::bind(&Fragment::getCharges, _1),
          charges,
          _2);
  return returnfunction;
}

void
PairPotential_Harmonic::setParametersToRandomInitialValues(
    const TrainingData &data)
{
  params[PairPotential_Harmonic::equilibrium_distance] = 3e+0*rand()/(double)RAND_MAX + .5;// 1.;
  params[PairPotential_Harmonic::spring_constant] = 1e+0*rand()/(double)RAND_MAX;// 0.2;
}

