/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * Please see the COPYING file or "Copyright notice" in builder.cpp for details.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>. 
 */

/*
 * PairPotential_Angle.cpp
 *
 *  Created on: Oct 11, 2012
 *      Author: heber
 */


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "PairPotential_Angle.hpp"

#include <boost/assign/list_of.hpp> // for 'map_list_of()'
#include <boost/bind.hpp>
#include <boost/lambda/lambda.hpp>
#include <string>

#include "CodePatterns/Assert.hpp"

#include "FunctionApproximation/Extractors.hpp"
#include "FunctionApproximation/TrainingData.hpp"
#include "Potentials/helpers.hpp"
#include "Potentials/ParticleTypeCheckers.hpp"

class Fragment;

// static definitions
const PairPotential_Angle::ParameterNames_t
PairPotential_Angle::ParameterNames =
      boost::assign::list_of<std::string>
      ("spring_constant")
      ("equilibrium_distance")
    ;
const std::string PairPotential_Angle::potential_token("harmonic_angle");

PairPotential_Angle::PairPotential_Angle() :
  EmpiricalPotential(),
  params(parameters_t(MAXPARAMS, 0.))
{
  // have some decent defaults for parameter_derivative checking
  params[spring_constant] = 1.;
  params[equilibrium_distance] = 0.1;
}

PairPotential_Angle::PairPotential_Angle(
    const ParticleTypes_t &_ParticleTypes
    ) :
  EmpiricalPotential(_ParticleTypes),
  params(parameters_t(MAXPARAMS, 0.))
{
  // have some decent defaults for parameter_derivative checking
  params[spring_constant] = 1.;
  params[equilibrium_distance] = 0.1;
}

PairPotential_Angle::PairPotential_Angle(
    const ParticleTypes_t &_ParticleTypes,
    const double _spring_constant,
    const double _equilibrium_distance) :
  EmpiricalPotential(_ParticleTypes),
  params(parameters_t(MAXPARAMS, 0.))
{
  params[spring_constant] = _spring_constant;
  params[equilibrium_distance] = _equilibrium_distance;
}

void PairPotential_Angle::setParameters(const parameters_t &_params)
{
  const size_t paramsDim = _params.size();
  ASSERT( paramsDim <= getParameterDimension(),
      "PairPotential_Angle::setParameters() - we need not more than "
      +toString(getParameterDimension())+" parameters.");
  for(size_t i=0;i<paramsDim;++i)
    params[i] = _params[i];

#ifndef NDEBUG
  parameters_t check_params(getParameters());
  check_params.resize(paramsDim); // truncate to same size
  ASSERT( check_params == _params,
      "PairPotential_Angle::setParameters() - failed, mismatch in to be set "
      +toString(_params)+" and set "+toString(check_params)+" params.");
#endif
}

PairPotential_Angle::result_t
PairPotential_Angle::function_theta(
    const double &r_ij,
    const double &r_jk,
    const double &r_ik
  ) const
{
//  Info info(__func__);
  const double angle = Helpers::pow(r_ij,2) + Helpers::pow(r_jk,2) - Helpers::pow(r_ik,2);
  const double divisor = 2.* r_ij * r_jk;

//  LOG(2, "DEBUG: cos(theta)= " << angle/divisor);
  if (divisor == 0.)
    return 0.;
  else
    return angle/divisor;
}

PairPotential_Angle::results_t
PairPotential_Angle::operator()(
    const arguments_t &arguments
    ) const
{
  ASSERT( arguments.size() == 3,
      "PairPotential_Angle::operator() - requires exactly three arguments.");
  ASSERT( ParticleTypeChecker::checkArgumentsAgainstParticleTypes(
      arguments, getParticleTypes()),
      "PairPotential_Angle::operator() - types don't match with ones in arguments.");
  const argument_t &r_ij = arguments[0]; // 01
  const argument_t &r_jk = arguments[2]; // 12
  const argument_t &r_ik = arguments[1]; // 02
  const result_t result =
      params[spring_constant]
             * Helpers::pow( function_theta(r_ij.distance, r_jk.distance, r_ik.distance) - params[equilibrium_distance], 2 );
  return std::vector<result_t>(1, result);
}

PairPotential_Angle::derivative_components_t
PairPotential_Angle::derivative(
    const arguments_t &arguments
    ) const
{
  ASSERT( arguments.size() == 3,
      "PairPotential_Angle::operator() - requires exactly three arguments.");
  ASSERT( ParticleTypeChecker::checkArgumentsAgainstParticleTypes(
      arguments, getParticleTypes()),
      "PairPotential_Angle::operator() - types don't match with ones in arguments.");
  derivative_components_t result;
  const argument_t &r_ij = arguments[0]; //01
  const argument_t &r_jk = arguments[2]; //12
  const argument_t &r_ik = arguments[1]; //02
  result.push_back( 2. * params[spring_constant] * ( function_theta(r_ij.distance, r_jk.distance, r_ik.distance) - params[equilibrium_distance]) );
  ASSERT( result.size() == 1,
      "PairPotential_Angle::operator() - we did not create exactly one component.");
  return result;
}

PairPotential_Angle::results_t
PairPotential_Angle::parameter_derivative(
    const arguments_t &arguments,
    const size_t index
    ) const
{
  ASSERT( arguments.size() == 3,
      "PairPotential_Angle::parameter_derivative() - requires exactly three arguments.");
  ASSERT( ParticleTypeChecker::checkArgumentsAgainstParticleTypes(
      arguments, getParticleTypes()),
      "PairPotential_Angle::operator() - types don't match with ones in arguments.");
  const argument_t &r_ij = arguments[0]; //01
  const argument_t &r_jk = arguments[2]; //12
  const argument_t &r_ik = arguments[1]; //02
  switch (index) {
    case spring_constant:
    {
      const result_t result =
                 Helpers::pow( function_theta(r_ij.distance, r_jk.distance, r_ik.distance) - params[equilibrium_distance], 2 );
      return std::vector<result_t>(1, result);
      break;
    }
    case equilibrium_distance:
    {
      const result_t result =
          -2. * params[spring_constant]
                 * ( function_theta(r_ij.distance, r_jk.distance, r_ik.distance) - params[equilibrium_distance]);
      return std::vector<result_t>(1, result);
      break;
    }
    default:
      ASSERT(0, "PairPotential_Angle::parameter_derivative() - derivative to unknown parameter desired.");
      break;
  }
}

FunctionModel::extractor_t
PairPotential_Angle::getFragmentSpecificExtractor() const
{
  Fragment::charges_t charges;
  charges.resize(getParticleTypes().size());
  std::transform(getParticleTypes().begin(), getParticleTypes().end(),
      charges.begin(), boost::lambda::_1);
  FunctionModel::extractor_t returnfunction =
      boost::bind(&Extractors::gatherDistancesFromFragment,
          boost::bind(&Fragment::getPositions, _1),
          boost::bind(&Fragment::getCharges, _1),
          charges,
          _2);
  return returnfunction;
}

void
PairPotential_Angle::setParametersToRandomInitialValues(
    const TrainingData &data)
{
  params[PairPotential_Angle::spring_constant] = 1e+0*rand()/(double)RAND_MAX;// 0.2;
  params[PairPotential_Angle::equilibrium_distance] = -0.3;//2e+0*rand()/(double)RAND_MAX - 1.;// 1.;
}

