/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * Please see the COPYING file or "Copyright notice" in builder.cpp for details.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>. 
 */

/*
 * ConstantPotential.cpp
 *
 *  Created on: May 09, 2013
 *      Author: heber
 */


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "ConstantPotential.hpp"

#include <boost/assign/list_of.hpp> // for 'map_list_of()'
#include <boost/bind.hpp>
#include <boost/lambda/lambda.hpp>
#include <cmath>
#include <string>

#include "CodePatterns/Assert.hpp"

#include "FunctionApproximation/Extractors.hpp"
#include "FunctionApproximation/TrainingData.hpp"
#include "Potentials/helpers.hpp"
#include "Potentials/ParticleTypeCheckers.hpp"

class Fragment;

// static definitions
const ConstantPotential::ParameterNames_t
ConstantPotential::ParameterNames =
      boost::assign::list_of<std::string>
      ("energy_offset") //
    ;
const std::string ConstantPotential::potential_token("constant");

ConstantPotential::ConstantPotential() :
    EmpiricalPotential(),
    params(parameters_t(MAXPARAMS, 0.))
{
  // have some decent defaults for parameter_derivative checking
  params[energy_offset] = 0.1;
}

ConstantPotential::ConstantPotential(
    const ParticleTypes_t &_ParticleTypes
    ) :
  EmpiricalPotential(_ParticleTypes),
  params(parameters_t(MAXPARAMS, 0.))
{
  // have some decent defaults for parameter_derivative checking
  params[energy_offset] = 0.1;
}

ConstantPotential::ConstantPotential(
    const ParticleTypes_t &_ParticleTypes,
    const double _energy_offset) :
  EmpiricalPotential(_ParticleTypes),
  params(parameters_t(MAXPARAMS, 0.))
{
  params[energy_offset] = _energy_offset;
}

void ConstantPotential::setParameters(const parameters_t &_params)
{
  const size_t paramsDim = _params.size();
  ASSERT( paramsDim <= getParameterDimension(),
      "ConstantPotential::setParameters() - we need not more than "
      +toString(getParameterDimension())+" parameters.");
  for(size_t i=0;i<paramsDim;++i)
    params[i] = _params[i];

#ifndef NDEBUG
  parameters_t check_params(getParameters());
  check_params.resize(paramsDim); // truncate to same size
  ASSERT( check_params == _params,
      "ConstantPotential::setParameters() - failed, mismatch in to be set "
      +toString(_params)+" and set "+toString(check_params)+" params.");
#endif
}

ConstantPotential::results_t
ConstantPotential::operator()(
    const arguments_t &arguments
    ) const
{
  ASSERT( arguments.size() == 0,
      "ConstantPotential::operator() - requires no argument.");
  ASSERT( ParticleTypeChecker::checkArgumentsAgainstParticleTypes(
      arguments, getParticleTypes()),
      "ConstantPotential::operator() - types don't match with ones in arguments.");
  const result_t result = params[energy_offset];
  return std::vector<result_t>(1, result);
}

ConstantPotential::derivative_components_t
ConstantPotential::derivative(
    const arguments_t &arguments
    ) const
{
  ASSERT( arguments.size() == 0,
      "ConstantPotential::operator() - requires no argument.");
  ASSERT( ParticleTypeChecker::checkArgumentsAgainstParticleTypes(
      arguments, getParticleTypes()),
      "ConstantPotential::operator() - types don't match with ones in arguments.");
  derivative_components_t result(1, 0.);
  return result;
}

ConstantPotential::results_t
ConstantPotential::parameter_derivative(
    const arguments_t &arguments,
    const size_t index
    ) const
{
  ASSERT( arguments.size() == 0,
      "ConstantPotential::parameter_derivative() - requires no argument.");
  ASSERT( ParticleTypeChecker::checkArgumentsAgainstParticleTypes(
      arguments, getParticleTypes()),
      "ConstantPotential::operator() - types don't match with ones in arguments.");
  switch (index) {
    case energy_offset:
    {
      // Maple result: 1
      const result_t result = +1.;
      return std::vector<result_t>(1, result);
      break;
    }
    default:
      break;
  }
  return std::vector<result_t>(1, 0.);
}

FunctionModel::extractor_t
ConstantPotential::getFragmentSpecificExtractor() const
{
  Fragment::charges_t charges;
  charges.resize(getParticleTypes().size());
  std::transform(getParticleTypes().begin(), getParticleTypes().end(),
      charges.begin(), boost::lambda::_1);
  FunctionModel::extractor_t returnfunction =
      boost::bind(&Extractors::gatherDistancesFromFragment,
          boost::bind(&Fragment::getPositions, _1),
          boost::bind(&Fragment::getCharges, _1),
          charges,
          _2);
  return returnfunction;
}

void
ConstantPotential::setParametersToRandomInitialValues(
    const TrainingData &data)
{
  params[ConstantPotential::energy_offset] =
      data.getTrainingOutputAverage()[0];// -1.;
}

