/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * ContinuousValueUnitTest.cpp
 *
 *  Created on: Sep 29, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ContinuousValueUnitTest.hpp"

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "Parser/ContinuousValue.hpp"

#include "CodePatterns/toString.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( ContinuousValueTest );


void ContinuousValueTest::setUp()
{
  // failing asserts should be thrown
  ASSERT_DO(Assert::Throw);

  ValidRange = new range<int>(1,4);
}

void ContinuousValueTest::tearDown()
{
  delete ValidRange;
}

/************************************ tests ***********************************/

/** Unit test for isValid.
 *
 */
void ContinuousValueTest::isValidValueTest()
{
  // create instance
  ContinuousValue<int> test(*ValidRange);

  // checking valid values
  for (int i=1; i<=4;++i)
    CPPUNIT_ASSERT_EQUAL(true, test.isValidValue(i));

  // checking invalid values
  for (int i=-10; i<=0;++i)
    CPPUNIT_ASSERT_EQUAL(false, test.isValidValue(i));
  for (int i=5; i<=0;++i)
    CPPUNIT_ASSERT_EQUAL(false, test.isValidValue(i));
}

/** Unit test for isValid.
 *
 */
void ContinuousValueTest::isValidTest()
{
  // create instance
  ContinuousValue<int> test(*ValidRange);

  // checking valid values
  for (int i=1; i<=4;++i)
    CPPUNIT_ASSERT_EQUAL(true, test.isValid(toString(i)));

  // checking invalid values
  for (int i=-10; i<=0;++i)
    CPPUNIT_ASSERT_EQUAL(false, test.isValid(toString(i)));
  for (int i=5; i<=0;++i)
    CPPUNIT_ASSERT_EQUAL(false, test.isValid(toString(i)));
}

/** Unit test for setting/getting valid range.
 *
 */
void ContinuousValueTest::setgetValidRangeTest()
{
  {
    // create instance
    ContinuousValue<int> test(*ValidRange);

    // extending range and checking
    for (int i=5; i<=6;++i)
      CPPUNIT_ASSERT_EQUAL(false, test.isValidValue(i));
    test.setValidRange(range<int>(1,6));
    for (int i=5; i<=6;++i)
      CPPUNIT_ASSERT_EQUAL(true, test.isValidValue(i));
  }

  {
    // create instance
    ContinuousValue<int> test(*ValidRange);

    // lowering range with set value
    test.setValue(4);
    CPPUNIT_ASSERT_EQUAL(true, test.ValueSet);
#ifndef NDEBUG
    std::cout << "The following Assert failures are intended and do not indicate a failure of the test." << std::endl;
    CPPUNIT_ASSERT_THROW(test.setValidRange(range<int>(1,3)), Assert::AssertionFailure);
#else
    test.setValidRange(range<int>(1,3));
#endif
#ifndef NDEBUG
    // no value is not set
    std::cout << "The following Assert failures are intended and do not indicate a failure of the test." << std::endl;
    CPPUNIT_ASSERT_THROW(test.get(), Assert::AssertionFailure);
#endif
    // value gets invalidated in either case
    CPPUNIT_ASSERT_EQUAL(false, test.ValueSet);
  }
}

/** Unit test for setValue and getValue.
 *
 */
void ContinuousValueTest::settergetterValueTest()
{
  // create instance
  ContinuousValue<int> test(*ValidRange);

  // unset calling of get, throws
#ifndef NDEBUG
  std::cout << "The following Assert failures are intended and do not indicate a failure of the test." << std::endl;
  CPPUNIT_ASSERT_THROW(test.getValue(), Assert::AssertionFailure);
#endif

  // setting invalid, throws
#ifndef NDEBUG
  std::cout << "The following Assert failures are intended and do not indicate a failure of the test." << std::endl;
  CPPUNIT_ASSERT_THROW(test.setValue(5), Assert::AssertionFailure);
#endif
#ifndef NDEBUG
  std::cout << "The following Assert failures are intended and do not indicate a failure of the test." << std::endl;
  CPPUNIT_ASSERT_THROW(test.setValue(0), Assert::AssertionFailure);
#endif

  CPPUNIT_ASSERT_EQUAL(false, test.ValueSet);
  // checking all valid ones
  for (int i=1; i<=4;++i) {
    test.setValue(i);
    CPPUNIT_ASSERT_EQUAL(true, test.ValueSet);
    CPPUNIT_ASSERT_EQUAL(i, test.getValue());
  }
}

/** Unit test for setters and getters.
 *
 */
void ContinuousValueTest::settergetterTest()
{
  // create instance
  ContinuousValue<int> test(*ValidRange);

  // unset calling of get, throws
#ifndef NDEBUG
  std::cout << "The following Assert failures are intended and do not indicate a failure of the test." << std::endl;
  CPPUNIT_ASSERT_THROW(test.get(), Assert::AssertionFailure);
#endif

  // setting invalid, throws
#ifndef NDEBUG
  std::cout << "The following Assert failures are intended and do not indicate a failure of the test." << std::endl;
  CPPUNIT_ASSERT_THROW(test.set(toString(5)), Assert::AssertionFailure);
#endif
#ifndef NDEBUG
  std::cout << "The following Assert failures are intended and do not indicate a failure of the test." << std::endl;
  CPPUNIT_ASSERT_THROW(test.set(toString(0)), Assert::AssertionFailure);
#endif

  // checking all valid ones
  for (int i=1; i<=4;++i) {
    test.set(toString(i));
    CPPUNIT_ASSERT_EQUAL(toString(i), test.get());
  }
}

