/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * AdjacencyListUnitTest.cpp
 *
 *  Created on: Oct 17, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "AdjacencyListUnitTest.hpp"

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <boost/lambda/lambda.hpp>
#include <iostream>
#include <iterator>
#include <iostream>
#include <vector>

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"

#include "Atom/atom.hpp"
#include "Descriptors/AtomDescriptor.hpp"
#include "Element/element.hpp"
#include "Element/periodentafel.hpp"
#include "Graph/AdjacencyList.hpp"
#include "molecule.hpp"
#include "World.hpp"
#include "WorldTime.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/
// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( AdjacencyListTest );

const std::string adjacencyfile ="\
1 2\n\
2 1 3\n\
3 2 4\n\
4 3 5\n\
5 4 6\n\
6 5 7\n\
7 6 8\n\
8 7 9\n\
9 8 10\n\
10 9\n";

const std::string wrongadjacencyfile1 ="\
1 2\n\
2 1\n\
4 5\n\
5 4 6\n\
6 5 7\n\
7 6 8\n\
8 7 9\n\
9 8 10\n\
10 9\n";

const std::string wrongadjacencyfile2 ="\
1 2\n\
2 1 3\n\
3 2 4\n\
4 3 5\n\
5 4 6\n\
6 5 7\n\
7 6 8\n\
8 7 9\n\
9 8 10\n\
10 9 11\n\
11 10\n";

const std::string wrongadjacencyfile3 ="\
1 2\n\
2 1 3\n\
3 2 4\n\
4 3 5\n\
5 4 7\n\
6\n\
7 5 8\n\
8 7 9\n\
9 8 10\n\
10 9\n";

// set up and tear down
void AdjacencyListTest::setUp()
{
  // failing asserts should be thrown
  ASSERT_DO(Assert::Throw);

  const element *hydrogen = World::getInstance().getPeriode()->FindElement(1);
  CPPUNIT_ASSERT(hydrogen != NULL);

  TestMolecule = World::getInstance().createMolecule();
  CPPUNIT_ASSERT(TestMolecule != NULL);
  for(int i=0;i<ATOM_COUNT;++i){
    atom *_atom = World::getInstance().createAtom();
    CPPUNIT_ASSERT(_atom != NULL);
    _atom->setType(hydrogen);
    TestMolecule->AddAtom(_atom);
    atoms.push_back(_atom);
    atomIds.push_back(_atom->getId());
  }
  CPPUNIT_ASSERT_EQUAL( (size_t)ATOM_COUNT, atoms.size());
  CPPUNIT_ASSERT_EQUAL( (size_t)ATOM_COUNT, atomIds.size());
  // create linear chain
  for(int i=0;i<ATOM_COUNT-1;++i)
    atoms[i]->addBond(WorldTime::getTime(), atoms[i+1]);

  // create map as it should be
  for(int i=0;i<ATOM_COUNT;++i) {
    if (i != 0) // first has only one bond
      comparisonMap.insert( std::make_pair(atomIds[i], atomIds[i-1]) );
    if (i != ATOM_COUNT-1) // last has only one bond
      comparisonMap.insert( std::make_pair(atomIds[i], atomIds[i+1]) );
  }
}

void AdjacencyListTest::tearDown()
{
  comparisonMap.clear();

  // destroy molecule and contained atoms
  removeAtomsinMolecule(TestMolecule);
  // destroy World
  World::purgeInstance();
//  logger::purgeInstance();
//  errorLogger::purgeInstance();
}

/** Unit tests for AdjacencyList::CreateMap().
 *
 */
void AdjacencyListTest::CreateMapTest()
{
  std::stringstream input;
  AdjacencyList FileAdjacency(input);
  AdjacencyList WorldAdjacency(atomIds);

  // check size (it's 8*2 + 2*1 = 18 keys)
  CPPUNIT_ASSERT_EQUAL( (size_t)18, WorldAdjacency.atombondmap.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)0, FileAdjacency.atombondmap.size() );

  // check equality
  CPPUNIT_ASSERT( comparisonMap.size() == WorldAdjacency.atombondmap.size() );
//  std::cout << "comparisonMap: " << comparisonMap << std::endl;
//  std::cout << "WorldAdjacency.atombondmap: " << WorldAdjacency.atombondmap << std::endl;
  CPPUNIT_ASSERT( comparisonMap == WorldAdjacency.atombondmap );

  // check non-equality: more
  comparisonMap.insert( std::make_pair( (atomId_t)10, (atomId_t)8) );
  CPPUNIT_ASSERT( comparisonMap != WorldAdjacency.atombondmap );
  comparisonMap.erase((atomId_t)10);

  // check non-equality: less
  comparisonMap.erase((atomId_t)9);
  CPPUNIT_ASSERT( comparisonMap != WorldAdjacency.atombondmap );
}

/** Unit tests for AdjacencyList::ParseIntoMap().
 *
 */
void AdjacencyListTest::ParseIntoMapTest()
{
  std::stringstream input(adjacencyfile);
  AdjacencyList FileAdjacency(input);
  AdjacencyList WorldAdjacency;

  // check size (it's 8*2 + 2*1 = 18 keys)
  CPPUNIT_ASSERT_EQUAL( (size_t)0, WorldAdjacency.atombondmap.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)18, FileAdjacency.atombondmap.size() );

  // check equality
  CPPUNIT_ASSERT( comparisonMap.size() == FileAdjacency.atombondmap.size() );
  CPPUNIT_ASSERT( comparisonMap == FileAdjacency.atombondmap );

  // check non-equality: more
  comparisonMap.insert( std::make_pair( (atomId_t)10, (atomId_t)8) );
  CPPUNIT_ASSERT( comparisonMap != FileAdjacency.atombondmap );
  comparisonMap.erase((atomId_t)10);

  // check non-equality: less
  comparisonMap.erase((atomId_t)9);
  CPPUNIT_ASSERT( comparisonMap != FileAdjacency.atombondmap );
}

/** Unit tests for AdjacencyList::StoreToFile().
 *
 */
void AdjacencyListTest::StoreToFileTest()
{
  std::stringstream input(adjacencyfile);
  AdjacencyList FileAdjacency(input);

  // check size (it's 8*2 + 2*1 = 18 keys)
  CPPUNIT_ASSERT_EQUAL( (size_t)18, FileAdjacency.atombondmap.size() );

  // store to file
  std::stringstream output;
  FileAdjacency.StoreToFile(output);
  CPPUNIT_ASSERT_EQUAL( input.str(), output.str() );
}

/** Unit tests for AdjacencyList::operator<().
 *
 */
void AdjacencyListTest::operatorGreaterLessTest()
{
  std::stringstream input(adjacencyfile);
  AdjacencyList FileAdjacency(input);
  AdjacencyList WorldAdjacency;

  // assert empty set is subset of some filled set (empty sets should always return true)
  CPPUNIT_ASSERT( FileAdjacency.atombondmap.size() != WorldAdjacency.atombondmap.size() );
  CPPUNIT_ASSERT( FileAdjacency.atombondmap != WorldAdjacency.atombondmap );
  CPPUNIT_ASSERT( WorldAdjacency < FileAdjacency );
  CPPUNIT_ASSERT( !(WorldAdjacency > FileAdjacency) );

  // parse in external map
  WorldAdjacency.CreateMap(atomIds);

  // assert equality after parsing
  CPPUNIT_ASSERT_EQUAL( FileAdjacency.atombondmap.size(), WorldAdjacency.atombondmap.size() );
  CPPUNIT_ASSERT( FileAdjacency.atombondmap == WorldAdjacency.atombondmap );
  CPPUNIT_ASSERT( WorldAdjacency < FileAdjacency );
  CPPUNIT_ASSERT( WorldAdjacency > FileAdjacency );

  // remove one entry from the world
  WorldAdjacency.atombondmap.erase((atomId_t)9);
  CPPUNIT_ASSERT( WorldAdjacency < FileAdjacency );
  CPPUNIT_ASSERT( !(WorldAdjacency > FileAdjacency) );
}

/** Unit tests for AdjacencyList::operator==().
 *
 */
void AdjacencyListTest::operatorEqualTest()
{
  std::stringstream input(adjacencyfile);
  AdjacencyList FileAdjacency(input);
  AdjacencyList WorldAdjacency;

  // assert equality before parsing (empty sets should always return true)
  CPPUNIT_ASSERT( FileAdjacency.atombondmap.size() != WorldAdjacency.atombondmap.size() );
  CPPUNIT_ASSERT( FileAdjacency.atombondmap != WorldAdjacency.atombondmap );
  CPPUNIT_ASSERT( WorldAdjacency != FileAdjacency );

  // parse in external map
  WorldAdjacency.CreateMap(atomIds);

  // assert equality after parsing
  CPPUNIT_ASSERT_EQUAL( FileAdjacency.atombondmap.size(), WorldAdjacency.atombondmap.size() );
  CPPUNIT_ASSERT( FileAdjacency.atombondmap == WorldAdjacency.atombondmap );
  CPPUNIT_ASSERT( WorldAdjacency == FileAdjacency );
}

/** Unit tests for AdjacencyList::operator()().
 *
 */
void AdjacencyListTest::CheckAgainstSubsetTest()
{
  AdjacencyList WorldAdjacency(atomIds);
  {
    // parse right
    std::stringstream input(adjacencyfile);
    AdjacencyList FileAdjacency(input);
    CPPUNIT_ASSERT( (WorldAdjacency > FileAdjacency) && (WorldAdjacency < FileAdjacency) );
  }
  {
    // parse wrong1 (more atoms in the world than in file, hence wrong)
    std::stringstream input(wrongadjacencyfile1);
    AdjacencyList FileAdjacency(input);
    CPPUNIT_ASSERT( !(WorldAdjacency < FileAdjacency) && (WorldAdjacency > FileAdjacency) );
  }
  {
    // remove third atom (index starts at 0) and test for equality then
    std::vector<atomId_t> validids;
    std::remove_copy_if(atomIds.begin(), atomIds.end(), std::back_inserter(validids), boost::lambda::_1 == (atomId_t)2);
    CPPUNIT_ASSERT_EQUAL( (size_t)ATOM_COUNT-1, validids.size() );
    AdjacencyList RestrictedWorldAdjacency(validids);
    // parse wrong1 (more atoms in the world than in file, hence wrong)
    std::stringstream input(wrongadjacencyfile1);
    AdjacencyList FileAdjacency(input);
    CPPUNIT_ASSERT( RestrictedWorldAdjacency == FileAdjacency );
  }
  {
    // parse wrong2 (there is no atom 10, but present in file. hence true)
    std::stringstream input(wrongadjacencyfile2);
    AdjacencyList FileAdjacency(input);
    CPPUNIT_ASSERT( !(WorldAdjacency > FileAdjacency) && (WorldAdjacency < FileAdjacency) );
  }
  {
    // parse wrong3 (6 is skipped in connection, hence neither is subset)
    std::stringstream input(wrongadjacencyfile3);
    AdjacencyList FileAdjacency(input);
//    WorldAdjacency.StoreToFile((std::ostream &)std::cout);
//    FileAdjacency.StoreToFile((std::ostream &)std::cout);
    CPPUNIT_ASSERT( WorldAdjacency != FileAdjacency );
  }
}
