/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * Fragment.cpp
 *
 *  Created on: Aug 8, 2012
 *      Author: heber
 */


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// include headers that implement a archive in simple text format
// otherwise BOOST_CLASS_EXPORT_IMPLEMENT has no effect
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>

#include "CodePatterns/MemDebug.hpp"

#include "Fragment.hpp"

#include <algorithm>
#include <cmath>
#include <iostream>
#include <limits>

#include "CodePatterns/Assert.hpp"

Fragment::Fragment()
{}

Fragment::Fragment(const positions_t &_positions, const charges_t &_charges)
{
  ASSERT( _positions.size() == _charges.size(),
      "Fragment::Fragment() - given charges and positions don't match in size.");
  positions_t::const_iterator piter = _positions.begin();
  charges_t::const_iterator citer = _charges.begin();
  for (;piter != _positions.end(); ++piter, ++citer)
    nuclei.push_back( make_pair( *piter, *citer) );
}


Fragment& Fragment::operator+=(const Fragment &other)
{
  for (nuclei_t::const_iterator iter = other.nuclei.begin();
      iter != other.nuclei.end(); ++iter) {
    if (!containsNuclei(*iter)) {
      nuclei.push_back(*iter);
    }
  }
  return *this;
}

Fragment& Fragment::operator=(const Fragment &other)
{
  if (this != &other) {
    nuclei.clear();
    nuclei = other.nuclei;
  }
  return *this;
}

Fragment& Fragment::operator-=(const Fragment &other)
{
  for (nuclei_t::const_iterator iter = other.nuclei.begin();
      iter != other.nuclei.end(); ++iter)
    removeNuclei(*iter);
  return *this;
}

bool Fragment::containsNuclei(const nucleus_t &n) const
{
  for (nuclei_t::const_iterator iter = nuclei.begin();
      iter != nuclei.end(); ++iter)
    if (Fragment::isPositionEqual(iter->first, n.first))
      return true;
  return false;
}

void Fragment::removeNuclei(const nucleus_t &n)
{
  for (nuclei_t::iterator iter = nuclei.begin();
      iter != nuclei.end(); ++iter)
    if (Fragment::isPositionEqual(iter->first, n.first)) {
      nuclei.erase(iter);
      break;
    }
}

Fragment::positions_t Fragment::getPositions() const
{
  positions_t positions;
  for (nuclei_t::const_iterator iter = nuclei.begin();
      iter != nuclei.end(); ++iter)
    positions.push_back(iter->first);
  return positions;
}

Fragment::charges_t Fragment::getCharges() const
{
  charges_t charges;
  for (nuclei_t::const_iterator iter = nuclei.begin();
      iter != nuclei.end(); ++iter)
    charges.push_back(iter->second);
  return charges;
}

Fragment::nucleus_t Fragment::createNucleus(const position_t &position, const double charge)
{
  return nucleus_t( make_pair( position, charge ) );
}

bool Fragment::isPositionEqual(const position_t &a, const position_t &b)
{
  bool status = true;
  for (size_t i=0;i<3;++i)
    status &= fabs(a[i] - b[i]) < std::numeric_limits<double>::epsilon();
  return status;
}

bool Fragment::operator==(const Fragment& other) const
{
  bool status = true;
  // compare sizes
  if (nuclei.size() != other.nuclei.size()) {
    return false;
  } else {
    // if equal, sort, and compare each nuclei
    Fragment::nuclei_t thisnuclei(nuclei);
    Fragment::nuclei_t othernuclei(other.nuclei);
    std::sort(thisnuclei.begin(), thisnuclei.end());
    std::sort(othernuclei.begin(), othernuclei.end());
    Fragment::nuclei_t::const_iterator iter = thisnuclei.begin();
    Fragment::nuclei_t::const_iterator oiter = othernuclei.begin();
    for (; iter != thisnuclei.end(); ++iter,++oiter)
      status &= (*iter == *oiter);
  }
  return status;
}

std::ostream & operator<<(std::ostream &ost, const Fragment &f)
{
  size_t NucleiCounter = 1;
  for (Fragment::nuclei_t::const_iterator iter = f.nuclei.begin();
      iter != f.nuclei.end(); ++iter)
    ost << "[" << NucleiCounter++ << "]:" << *iter << ", ";
  return ost;
}

std::ostream & operator<<(std::ostream &ost, const Fragment::nucleus_t &n)
{
  ost << n.first << " with charge " << n.second;
  return ost;
}

bool operator==(const Fragment::nucleus_t &a, const Fragment::nucleus_t &b)
{
  bool status = true;
  // check positions
  status &= Fragment::isPositionEqual(a.first, b.first);
  status &= fabs(a.second - b.second) < std::numeric_limits<double>::epsilon();
  return status;
}

template<> Fragment ZeroInstance<Fragment>()
{
  Fragment::positions_t positions;
  Fragment::charges_t charges;
  Fragment returnvalue(positions, charges);
  return returnvalue;
}


// we need to explicitly instantiate the serialization functions
BOOST_CLASS_EXPORT_IMPLEMENT(Fragment)
