/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * OperationQueueUnitTest.cpp
 *
 *  Created on: Apr 24, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "OperationQueueUnitTest.hpp"

#include "CodePatterns/Assert.hpp"

#include "Operations/OperationQueue.hpp"
#include "stubs/AsyncOperationStub.hpp"
#include "WorkerAddress.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( OperationQueueTest );

void OperationQueueTest::setUp()
{
  // Throw assertions
  ASSERT_DO(Assert::Throw);

  queue = new OperationQueue;
  conn = new Connection(io_service);
  address = new WorkerAddress("", ""); // address is invalid in sense of OperationQueue to prevent connect

  op = new AsyncOperationStub(std::string("AsyncOperationStub"), *conn);
  otherop = new AsyncOperationStub(std::string("AsyncOperationStub"), *conn);
}


void OperationQueueTest::tearDown()
{
  delete conn;
  delete queue;
  delete address;

  // if op and otherop are pushed, then they are NULL, hence we may call delete on them
  delete op;
  delete otherop;
}

/** UnitTest for findOperation
 */
void OperationQueueTest::findTest()
{
  AsyncOperation *backup = op; // we need backup as op is NULL'ed
  AsyncOperation *otherbackup = otherop; // we need otherbackup as otherop is NULL'ed
  // queue is empty
  CPPUNIT_ASSERT( queue->findOperation(op) == queue->queue.end() );
  CPPUNIT_ASSERT( queue->findOperation(otherop) == queue->queue.end() );

  // insert ptr
  queue->push_back(op, *address);
  CPPUNIT_ASSERT( queue->findOperation(backup) != queue->queue.end() );
  CPPUNIT_ASSERT( queue->findOperation(otherop) == queue->queue.end() );

  // insert otherptr
  queue->push_back(otherop, *address);
  CPPUNIT_ASSERT( queue->findOperation(backup) != queue->queue.end() );
  CPPUNIT_ASSERT( queue->findOperation(otherbackup) != queue->queue.end() );

  std::cout << "The following warning's do not indicate a failure." << std::endl;
}

/** UnitTest for push_back
 */
void OperationQueueTest::push_backTest()
{
  AsyncOperation *backup = op; // we need backup as op is NULL'ed
  queue->push_back(op,* address);
  CPPUNIT_ASSERT_EQUAL( (size_t)1, queue->queue.size() );
  CPPUNIT_ASSERT_EQUAL( backup, queue->queue.begin()->get() );

  queue->push_back(otherop, *address);
  CPPUNIT_ASSERT_EQUAL( (size_t)2, queue->queue.size() );
  CPPUNIT_ASSERT_EQUAL( backup, queue->queue.begin()->get() );

  std::cout << "The following warning's do not indicate a failure." << std::endl;
}

/** UnitTest for remove
 */
void OperationQueueTest::removeTest()
{
  // insert op
  AsyncOperation *backup = op; // we need backup as op is NULL'ed
  queue->push_back(op, *address);
  CPPUNIT_ASSERT( queue->queue.begin() != queue->queue.end() );

  // remove op
  queue->remove(backup, queue);
  CPPUNIT_ASSERT( queue->queue.begin() == queue->queue.end() );
}

/** UnitTest whether observer is correctly working.
 *
 */
void OperationQueueTest::observerTest()
{
  // sign on to our op
  AsyncOperation *backup = op; // we need backup as op is NULL'ed

  // place into queue
  queue->push_back(op, *address);

  // and signal to queue it is done
  queue->update(backup);
  backup = NULL;  // should be deleted from this line on

  // check queue is now empty
  CPPUNIT_ASSERT( queue->queue.begin() == queue->queue.end() );
}
