/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * OperationQueue.cpp
 *
 *  Created on: Apr 24, 2012
 *      Author: heber
 */


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// boost asio needs specific operator new
#include <boost/asio.hpp>

#include "CodePatterns/MemDebug.hpp"

#include <boost/bind.hpp>
#include <boost/lambda/lambda.hpp>
#include <string>

#include "CodePatterns/Log.hpp"
#include "CodePatterns/Observer/Observer.hpp"

#include "Operations/AsyncOperation.hpp"
#include "Operations/OperationQueue.hpp"
#include "WorkerAddress.hpp"

OperationQueue::OperationQueue_t::iterator OperationQueue::findOperation(AsyncOperation *op)
{
  OperationQueue_t::iterator iter =
      std::find_if(queue.begin(), queue.end(),
          boost::bind(&AsyncOp_ptr::get, boost::lambda::_1) == op);
  return iter;
}

void OperationQueue::push_back(AsyncOperation *&op, const WorkerAddress &address)
{
  if (op != NULL) {
    AsyncOp_ptr ptr(op); // this always prevents memory loss
    ptr->signOn(this);
    queue.push_back( ptr );
    op = NULL;
    // only start operation when address is valid
    if ((!address.host.empty()) && (!address.service.empty()))
      (*ptr)(address.host, address.service);
  } else {
    ELOG(1, "Given operation pointer is NULL.");
  }
}

void OperationQueue::remove(AsyncOperation *op, Observer *observer)
{
  if (op != NULL) {
    OperationQueue_t::iterator iter = findOperation(op);
    if (iter != queue.end()) {
      // sign off and remove op
      if (observer != NULL)
        op->signOff(observer);
      queue.erase(iter);
    } else {
      ELOG(1, "Could not find Operation " << op->getName() << " in operation's queue.");
    }
  } else {
    ELOG(1, "Given operation pointer is NULL.");
  }
}

void OperationQueue::update(Observable *publisher)
{
  AsyncOperation *op = static_cast<AsyncOperation *>(publisher);
  if (op != NULL) {
    LOG(1, "INFO: We are note notified that " << op->getName() << " is done, removing ...");
    // remove from queue
    remove(op, this);
  }
}

void OperationQueue::recieveNotification(Observable *publisher, Notification_ptr notification)
{}

void OperationQueue::subjectKilled(Observable *publisher)
{
  AsyncOperation *op = static_cast<AsyncOperation *>(publisher);
  if (op != NULL) {
    ELOG(2, "DEBUG: AsyncOperation at " << publisher << " got killed before being done?");
    // remove from queue
    remove(op, this);
  }
}

