/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * MPQCJob.cpp
 *
 *  Created on: Jul 10, 2012
 *      Author: heber
 */


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// include headers that implement a archive in simple text format
// otherwise BOOST_CLASS_EXPORT_IMPLEMENT has no effect
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>

#include "CodePatterns/MemDebug.hpp"

#include "MPQCJob.hpp"

#include <sstream>

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"


MPQCJob::MPQCJob(
    const JobId_t _JobId,
    const std::string &_inputfile,
    const double _begin[3],
    const double _end[3],
    const int _level) :
  FragmentJob(_JobId),
  DoLongrange(MPQCData::DontSampleDensity),
  DoValenceOnly(MPQCData::DoSampleValenceOnly),
  inputfile(_inputfile),
  grid(_begin, _end, _level)
{
  ASSERT( (_end[0] == _end[1]) && (_end[0] == _end[2]),
      "MPQCJob::MPQCJob() - box needs to be cubic.");
}

MPQCJob::MPQCJob() :
  FragmentJob(JobId::IllegalJob),
  DoLongrange(MPQCData::DontSampleDensity),
  DoValenceOnly(MPQCData::DoSampleValenceOnly)
{}

MPQCJob::~MPQCJob()
{}

FragmentResult::ptr MPQCJob::Work()
{
  // instantiate empty data
  const MPQCData data;
  std::stringstream returnstream;
  boost::archive::text_oarchive oa(returnstream);
  oa << data;

  // create the result
  FragmentResult::ptr ptr( new FragmentResult(getId(), returnstream.str()) );
  if (data.desired_accuracy != 0.) {
    ptr->exitflag = data.accuracy < data.desired_accuracy ? 0 : 1;
    if (ptr->exitflag != 0)
      ELOG(1, "Job #" << ptr->getId() << " failed to reach desired accuracy.");
  } else {
    LOG(3, "INFO: No precision returned from MPQC job, not checking.");
  }

  return ptr;
}

// we need to explicitly instantiate the serialization functions as
// its is only serialized through its base class FragmentJob
BOOST_CLASS_EXPORT_IMPLEMENT(MPQCJob)
