/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2015 Frederik Heber. All rights reserved.
 *
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * QtObservedMolecule.cpp
 *
 *  Created on: Oct 28, 2015
 *      Author: heber
 */


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "QtObservedMolecule.hpp"

#include "UIElements/Qt4/InstanceBoard/QtObservedInstanceBoard.hpp"

#include "CodePatterns/MemDebug.hpp"
#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"

#include <boost/assign.hpp>
#include <boost/bind.hpp>

#include "UIElements/Qt4/InstanceBoard/ObservedValue_wCallback.hpp"
#include "UIElements/Qt4/InstanceBoard/ObservedValue_UpdateAtoms.hpp"

#include "Atom/atom.hpp"
#include "Descriptors/MoleculeIdDescriptor.hpp"
#include "World.hpp"

using namespace boost::assign;

static const Observable::channels_t getAllObservedChannels()
{
  Observable::channels_t channels;
  channels +=
      molecule::AtomInserted,
      molecule::AtomMoved,
      molecule::AtomRemoved,
      molecule::FormulaChanged,
      molecule::IndexChanged,
      molecule::MoleculeNameChanged,
      molecule::BoundingBoxChanged,
      molecule::SelectionChanged;
  return channels;
}

static const Observable::channels_t getAllAtomCountChannels()
{
  Observable::channels_t channels;
  channels +=
      molecule::AtomInserted,
      molecule::AtomRemoved;
  return channels;
}

static const Observable::channels_t getAllCenterChannels()
{
  Observable::channels_t channels;
  channels +=
      molecule::AtomInserted,
      molecule::AtomMoved,
      molecule::AtomRemoved;
  return channels;
}

// static instances
const Observable::channels_t QtObservedMolecule::AtomCountChannels(getAllAtomCountChannels());
const Observable::channels_t QtObservedMolecule::BondCountChannels(getAllAtomCountChannels());
const Observable::channels_t QtObservedMolecule::BoundingBoxChannels(1, molecule::BoundingBoxChanged);
const Observable::channels_t QtObservedMolecule::FormulaStringChannels(1, molecule::FormulaChanged);
const Observable::channels_t QtObservedMolecule::CenterChannels(getAllCenterChannels());
const Observable::channels_t QtObservedMolecule::IndexChannels(1, molecule::IndexChanged);
const Observable::channels_t QtObservedMolecule::NameChannels(1, molecule::MoleculeNameChanged);
const Observable::channels_t QtObservedMolecule::NonHydrogenCountChannels(1, molecule::FormulaChanged);
const Observable::channels_t QtObservedMolecule::SelectedChannels(1, molecule::SelectionChanged);

QtObservedMolecule::QtObservedMolecule(
    const ObservedValues_t &_ObservedValues,
    QtObservedInstanceBoard &_board,
    QWidget * _parent) :
  QWidget(_parent),
  Observer("QtObservedMolecule"),
  subjectKilledCount(0),
  AllsignedOnChannels(getAllObservedChannels().size()),
  signedOffChannels(0),
  owner(NULL),
  board(_board),
  BoardIsGone(false),
  ObservedValues(_ObservedValues)
{
  // activating Observer is done by ObservedValueContainer when it's prepared
}

QtObservedMolecule::~QtObservedMolecule()
{
  boost::any_cast<ObservedValue_wCallback<moleculeId_t> *>(ObservedValues[MolIndex])->noteCallBackIsGone();
  boost::any_cast<ObservedValue_wCallback<int, moleculeId_t> *>(ObservedValues[AtomCount])->noteCallBackIsGone();
  boost::any_cast<ObservedValue_wCallback<int, moleculeId_t> *>(ObservedValues[BondCount])->noteCallBackIsGone();
  boost::any_cast<ObservedValue_wCallback<molecule::BoundingBoxInfo, moleculeId_t> *>(ObservedValues[BoundingBox])->noteCallBackIsGone();
  boost::any_cast<ObservedValue_wCallback<std::string, moleculeId_t> *>(ObservedValues[FormulaString])->noteCallBackIsGone();
  boost::any_cast<ObservedValue_wCallback<Vector, moleculeId_t> *>(ObservedValues[MolCenter])->noteCallBackIsGone();
  boost::any_cast<ObservedValue_wCallback<std::string, moleculeId_t> *>(ObservedValues[MolName])->noteCallBackIsGone();
  boost::any_cast<ObservedValue_wCallback<int, moleculeId_t> *>(ObservedValues[NonHydrogenCount])->noteCallBackIsGone();
  boost::any_cast<ObservedValue_wCallback<bool, moleculeId_t> *>(ObservedValues[MolSelected])->noteCallBackIsGone();

  deactivateObserver();
}

void QtObservedMolecule::deactivateObserver()
{
  if (owner != NULL) {
    Observable::channels_t channels = getAllObservedChannels();
    for (Observable::channels_t::const_iterator iter = channels.begin();
        iter != channels.end(); ++iter)
      owner->signOff(this, *iter);
    owner = NULL;
    signedOffChannels = AllsignedOnChannels;
    if (!BoardIsGone)
      board.markObservedMoleculeAsDisconnected(getMolIndex());
  }
}

void QtObservedMolecule::activateObserver()
{
  // sign on as observer (obtain non-const instance before)
  const molecule * const _molecule = getMolecule(getMolIndex());
  if (_molecule != NULL) {
    Observable::channels_t channels = getAllObservedChannels();
    owner = static_cast<const Observable *>(_molecule);
    for (Observable::channels_t::const_iterator iter = channels.begin();
        iter != channels.end(); ++iter)
      owner->signOn(this, *iter);
    if (!BoardIsGone)
      board.markObservedMoleculeAsConnected(getMolIndex());
  } else
    signedOffChannels = AllsignedOnChannels;
}

void QtObservedMolecule::update(Observable *publisher)
{
  ASSERT(0,
      "QtObservedMolecule::update() - general update from unexpected source.");
}

void QtObservedMolecule::subjectKilled(Observable *publisher)
{
  ++signedOffChannels;

  if (signedOffChannels == AllsignedOnChannels) {
    // remove owner: no more signOff needed
    owner = NULL;

    emit moleculeRemoved();

    if (!BoardIsGone) {
      board.markObservedMoleculeAsDisconnected(getMolIndex());
      board.markObservedMoleculeForErase(getMolIndex());
    }
  }
}

void QtObservedMolecule::recieveNotification(Observable *publisher, Notification_ptr notification)
{
  const molecule * const _molecule = getMolecule(getMolIndex());
  // when molecule is NULL we will soon get destroyed anyway
  if (_molecule == NULL)
    return;
  if (publisher == dynamic_cast<const Observable*>(_molecule)){
    // notification from atom
#ifdef LOG_OBSERVER
    observerLog().addMessage() << "++ Update of Observer "<< observerLog().getName(static_cast<Observer *>(this))
          << " received notification from molecule " << getMolIndex() << " for channel "
          << notification->getChannelNo() << ".";
#endif
    switch (notification->getChannelNo()) {
      case molecule::AtomInserted:
      {
        const atomId_t _id = _molecule->lastChangedAtomId();
        emit atomcountChanged();
        emit atomInserted(_id);
        emit bondcountChanged();
        emit boundingboxChanged();
        emit centerChanged();
        emit tesselationhullChanged();
        break;
      }
      case molecule::AtomMoved:
      {
        emit boundingboxChanged();
        emit centerChanged();
        emit tesselationhullChanged();
        break;
      }
      case molecule::AtomRemoved:
      {
        const atomId_t _id = _molecule->lastChangedAtomId();
        emit atomcountChanged();
        emit atomRemoved(_id);
        emit bondcountChanged();
        emit boundingboxChanged();
        emit centerChanged();
        emit tesselationhullChanged();
        break;
      }
      case molecule::BoundingBoxChanged:
      {
  #ifdef LOG_OBSERVER
        observerLog().addMessage() << "++ Observer " << observerLog().getName(static_cast<Observer *>(this)) << " received notification that bounding box has changed.";
  #endif
        emit tesselationhullChanged();
        emit boundingboxChanged();
        break;
      }
      case molecule::FormulaChanged:
      {
        emit formulaChanged();
        emit nononhydrogenChanged();
        break;
      }
      case molecule::IndexChanged:
      {
  #ifdef LOG_OBSERVER
        const atomId_t _id = _molecule->lastChangedAtomId();
        observerLog().addMessage() << "++ Observer " << observerLog().getName(static_cast<Observer *>(this)) << " received notification that atom "+toString(_id)+"'s index has changed.";
  #endif
        emit indexChanged();
        break;
      }
      case molecule::MoleculeNameChanged:
      {
  #ifdef LOG_OBSERVER
        observerLog().addMessage() << "++ Observer " << observerLog().getName(static_cast<Observer *>(this)) << " received notification that name has changed.";
  #endif
        emit nameChanged();
        break;
      }
      case molecule::SelectionChanged:
      {
  #ifdef LOG_OBSERVER
        observerLog().addMessage() << "++ Observer " << observerLog().getName(static_cast<Observer *>(this)) << " received notification that selected has changed.";
  #endif
        emit selectedChanged();
        break;
      }
      default:
        break;
    }
  }
}

const molecule * const QtObservedMolecule::getMolecule(const moleculeId_t _id)
{
  const molecule * const mol = const_cast<const World &>(World::getInstance()).
      getMolecule(MoleculeById(_id));
  return mol;
}

static molecule::BoundingBoxInfo initBoundingBox()
{
  molecule::BoundingBoxInfo info;
  info.position = zeroVec;
  info.radius = 0.;
  return info;
}

void QtObservedMolecule::initObservedValues(
    ObservedValues_t &_ObservedValues,
    const moleculeId_t _molid,
    const molecule * const _molref,
    const boost::function<void(const moleculeId_t)> &_subjectKilled)
{
  /* This is an old note from when the code was still part of cstor's initializer body.
   * TODO: Probably does not apply anymore but has not yet been tested.
   *
   * We must not use boost::cref(this) as "this" has not been properly constructed and seemingly
   * boost::cref tries to do some magic to grasp the inheritance hierarchy which fails because
   * the class has not been fully constructed yet. "This" itself seems to be working fine.
   */

  ASSERT( _ObservedValues.size() == MAX_ObservedTypes,
      "QtObservedMolecule::initObservedValues() - given ObservedValues has not correct size.");

  // fill ObservedValues: index first
  const boost::function<moleculeId_t ()> MolIndexUpdater(
      boost::bind(&QtObservedMolecule::updateIndex));

  ObservedValue_wCallback<moleculeId_t> * const IndexObservable =
      new ObservedValue_wCallback<moleculeId_t>(
        _molref,
        MolIndexUpdater,
        "MoleculeIndex_"+toString(_molid),
        _molid,
        IndexChannels,
        _subjectKilled);
  _ObservedValues[MolIndex] = IndexObservable;

  const boost::function<const moleculeId_t ()> MolIndexGetter =
      boost::bind(&ObservedValue_wCallback<moleculeId_t>::get,
          IndexObservable);

  // fill ObservedValues: then all the other that need index
  const boost::function<int ()> AtomCountUpdater(
      boost::bind(&QtObservedMolecule::updateAtomCount, MolIndexGetter));
  const boost::function<int ()> BondCountUpdater(
      boost::bind(&QtObservedMolecule::updateBondCount, MolIndexGetter));
  const boost::function<Vector ()> MolCenterUpdater(
      boost::bind(&QtObservedMolecule::updateCenter, MolIndexGetter));
  const boost::function<std::string ()> MolFormulaUpdater(
      boost::bind(&QtObservedMolecule::updateFormulaString, MolIndexGetter));
  const boost::function<std::string ()> MolNameUpdater(
      boost::bind(&QtObservedMolecule::updateName, MolIndexGetter));
  const boost::function<int ()> NonHydrogenCountUpdater(
      boost::bind(&QtObservedMolecule::updateNonHydrogenCount, MolIndexGetter));
  const boost::function<molecule::BoundingBoxInfo ()> BoundingBoxUpdater(
      boost::bind(&QtObservedMolecule::updateBoundingBox, MolIndexGetter));
  const boost::function<bool ()> SelectedUpdater(
      boost::bind(&QtObservedMolecule::updateSelected, MolIndexGetter));

  _ObservedValues[AtomCount] = new ObservedValue_wCallback<int, moleculeId_t>(
      _molref,
      AtomCountUpdater,
      "MoleculeAtomCount_"+toString(_molid),
      AtomCountUpdater(),
      AtomCountChannels,
      _subjectKilled,
      MolIndexGetter);
  _ObservedValues[BondCount] = new ObservedValue_wCallback<int, moleculeId_t>(
      _molref,
      BondCountUpdater,
      "MoleculeBondCount_"+toString(_molid),
      BondCountUpdater(),
      BondCountChannels,
      _subjectKilled,
      MolIndexGetter);
  _ObservedValues[BoundingBox] = new ObservedValue_wCallback<molecule::BoundingBoxInfo, moleculeId_t>(
      _molref,
      BoundingBoxUpdater,
      "MoleculeBoundingBox_"+toString(_molid),
      initBoundingBox(),
      BoundingBoxChannels,
      _subjectKilled,
      MolIndexGetter);
  _ObservedValues[FormulaString] = new ObservedValue_wCallback<std::string, moleculeId_t>(
      _molref,
      MolFormulaUpdater,
      "MoleculeFormula_"+toString(_molid),
      MolFormulaUpdater(),
      FormulaStringChannels,
      _subjectKilled,
      MolIndexGetter);
  _ObservedValues[MolCenter] = new ObservedValue_wCallback<Vector, moleculeId_t>(
      _molref,
      MolCenterUpdater,
      "MoleculeCenter_"+toString(_molid),
      MolCenterUpdater(),
      CenterChannels,
      _subjectKilled,
      MolIndexGetter);
  _ObservedValues[MolName] = new ObservedValue_wCallback<std::string, moleculeId_t>(
      _molref,
      MolNameUpdater,
      "MoleculeName_"+toString(_molid),
      MolNameUpdater(),
      NameChannels,
      _subjectKilled,
      MolIndexGetter);
  _ObservedValues[NonHydrogenCount] = new ObservedValue_wCallback<int, moleculeId_t>(
      _molref,
      NonHydrogenCountUpdater,
      "MoleculeNonHydrogenCount_"+toString(_molid),
      NonHydrogenCountUpdater(),
      NonHydrogenCountChannels,
      _subjectKilled,
      MolIndexGetter);
  _ObservedValues[MolSelected] = new ObservedValue_wCallback<bool, moleculeId_t>(
      _molref,
      SelectedUpdater,
      "MoleculeSelected_"+toString(_molid),
      SelectedUpdater(),
      SelectedChannels,
      _subjectKilled,
      MolIndexGetter);
}

void QtObservedMolecule::destroyObservedValues(
    std::vector<boost::any> &_ObservedValues)
{
  delete boost::any_cast<ObservedValue_wCallback<moleculeId_t> *>(_ObservedValues[MolIndex]);
  delete boost::any_cast<ObservedValue_wCallback<int, moleculeId_t> *>(_ObservedValues[AtomCount]);
  delete boost::any_cast<ObservedValue_wCallback<int, moleculeId_t> *>(_ObservedValues[BondCount]);
  delete boost::any_cast<ObservedValue_wCallback<molecule::BoundingBoxInfo, moleculeId_t> *>(_ObservedValues[BoundingBox]);
  delete boost::any_cast<ObservedValue_wCallback<std::string, moleculeId_t> *>(_ObservedValues[FormulaString]);
  delete boost::any_cast<ObservedValue_wCallback<Vector, moleculeId_t> *>(_ObservedValues[MolCenter]);
  delete boost::any_cast<ObservedValue_wCallback<std::string, moleculeId_t> *>(_ObservedValues[MolName]);
  delete boost::any_cast<ObservedValue_wCallback<int, moleculeId_t> *>(_ObservedValues[NonHydrogenCount]);
  delete boost::any_cast<ObservedValue_wCallback<bool, moleculeId_t> *>(_ObservedValues[MolSelected]);
  _ObservedValues.clear();
}

int QtObservedMolecule::updateAtomCount(
    const boost::function<const moleculeId_t ()> &_getMolIndex)
{
  const molecule * const mol = getMolecule(_getMolIndex());
  if (mol != NULL)
    return mol->getAtomCount();
  else
    return (int)0;
}

int QtObservedMolecule::updateBondCount(
    const boost::function<const moleculeId_t ()> &_getMolIndex)
{
  const molecule * const mol = getMolecule(_getMolIndex());
  if (mol != NULL)
    return mol->getBondCount();
  else
    return (int)0;
}

molecule::BoundingBoxInfo QtObservedMolecule::updateBoundingBox(
    const boost::function<const moleculeId_t ()> &_getMolIndex)
{
  const molecule * const mol = getMolecule(_getMolIndex());
  if (mol != NULL)
    return mol->getBoundingBox();
  else
    return molecule::BoundingBoxInfo();
}

std::string QtObservedMolecule::updateFormulaString(
    const boost::function<const moleculeId_t ()> &_getMolIndex)
{
  const molecule * const mol = getMolecule(_getMolIndex());
  if (mol != NULL)
    return mol->getFormula().toString();
  else
    return std::string("");
}

Vector QtObservedMolecule::updateCenter(
    const boost::function<const moleculeId_t ()> &_getMolIndex)
{
  const molecule * const mol = getMolecule(_getMolIndex());
  if (mol != NULL)
    return mol->DetermineCenterOfAll();
  else
    return zeroVec;
}

moleculeId_t QtObservedMolecule::updateIndex()
{
  return const_cast<const World &>(World::getInstance()).lastChangedMolId();
}

std::string QtObservedMolecule::updateName(
    const boost::function<const moleculeId_t ()> &_getMolIndex)
{
  const molecule * const mol = getMolecule(_getMolIndex());
  if (mol != NULL)
    return mol->getName();
  else
    return std::string("");
}

int QtObservedMolecule::updateNonHydrogenCount(
    const boost::function<const moleculeId_t ()> &_getMolIndex)
{
  const molecule * const mol = getMolecule(_getMolIndex());
  if (mol != NULL)
    return mol->getNoNonHydrogen();
  else
    return (int)0;
}

bool QtObservedMolecule::updateSelected(
    const boost::function<const moleculeId_t ()> &_getMolIndex)
{
  const molecule * const mol = getMolecule(_getMolIndex());
  if (mol != NULL)
    return mol->getSelected();
  else
    return false;
}

const int& QtObservedMolecule::getAtomCount() const
{
  return boost::any_cast<ObservedValue_wCallback<int, moleculeId_t> *>(ObservedValues[AtomCount])->get();
}

const int& QtObservedMolecule::getBondCount() const
{
  return boost::any_cast<ObservedValue_wCallback<int, moleculeId_t> *>(ObservedValues[BondCount])->get();
}

const std::string& QtObservedMolecule::getMolFormula() const
{
  return boost::any_cast<ObservedValue_wCallback<std::string, moleculeId_t> *>(ObservedValues[FormulaString])->get();
}

const Vector& QtObservedMolecule::getMolCenter() const
{
  return boost::any_cast<ObservedValue_wCallback<Vector, moleculeId_t> *>(ObservedValues[MolCenter])->get();
}

const moleculeId_t& QtObservedMolecule::getMolIndex() const
{
  return boost::any_cast<ObservedValue_wCallback<moleculeId_t> *>(ObservedValues[MolIndex])->get();
}

const std::string& QtObservedMolecule::getMolName() const
{
  return boost::any_cast<ObservedValue_wCallback<std::string, moleculeId_t> *>(ObservedValues[MolName])->get();
}

const int& QtObservedMolecule::getNonHydrogenCount() const
{
  return boost::any_cast<ObservedValue_wCallback<int, moleculeId_t> *>(ObservedValues[NonHydrogenCount])->get();
}

const molecule::BoundingBoxInfo& QtObservedMolecule::getBoundingBox() const
{
  return boost::any_cast<ObservedValue_wCallback<molecule::BoundingBoxInfo, moleculeId_t> *>(ObservedValues[BoundingBox])->get();
}

const bool& QtObservedMolecule::getMolSelected() const
{
  return boost::any_cast<ObservedValue_wCallback<bool, moleculeId_t> *>(ObservedValues[MolSelected])->get();
}
