/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * ObserverStub.cpp
 *
 *  Created on: Jan 19, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cppunit/extensions/HelperMacros.h>

#include "Assert.hpp"

#include "ObserverStub.hpp"

/************ UpdateCountObserver **************/

UpdateCountObserver::UpdateCountObserver() :
  Observer("UpdateCountObserver"),
  updates(0)
{};

UpdateCountObserver::~UpdateCountObserver()
{}

void UpdateCountObserver::update(Observable *publisher){
  updates++;
}

void UpdateCountObserver::subjectKilled(Observable *publisher) {
}

/*************** SimpleObservable **************/

SimpleObservable::SimpleObservable() :
  Observable("SimpleObservable")
{}

void SimpleObservable::changeMethod() {
  OBSERVE;
  int i = 0;
  i++;
}

/************** CallObservable *****************/

CallObservable::CallObservable() :
  Observable("CallObservable")
{}

void CallObservable::changeMethod1() {
  OBSERVE;
  int i = 0;
  i++;
}

void CallObservable::changeMethod2() {
  OBSERVE;
  int i = 0;
  i++;
  changeMethod1();
}

/************* BlockObservable *****************/

BlockObservable::BlockObservable() :
  Observable("BlockObservable")
{}

void BlockObservable::changeMethod1(){
  OBSERVE;
  // test if we report correctly as blocked
  CPPUNIT_ASSERT(isBlocked());
}

void BlockObservable::changeMethod2(){
  OBSERVE;
  internalMethod1();
  internalMethod2();
}

void BlockObservable::internalMethod1(){
  // we did not block, but our caller did...
  // see if this is found
  CPPUNIT_ASSERT(isBlocked());
}

void BlockObservable::internalMethod2(){
  OBSERVE;
  // Both this method and the caller do block
  // Does the reporting still work as expected?
  CPPUNIT_ASSERT(isBlocked());
}

void BlockObservable::noChangeMethod(){
  // No Block introduced here
  // reported correctely?
  CPPUNIT_ASSERT(!isBlocked());
}

/*************** SuperObservable ***************/

SuperObservable::SuperObservable():
  Observable("SuperObservable")
{
  subObservable = new SimpleObservable();
  subObservable->signOn(this);
}
SuperObservable::~SuperObservable(){
  delete subObservable;
}
void SuperObservable::changeMethod() {
  OBSERVE;
  int i = 0;
  i++;
  subObservable->changeMethod();
}

/************* NotificationObservable **********/

NotificationObservable::NotificationObservable() :
  Observable("NotificationObservable")
{
  NotificationChannels = new Channels(this);
  NotificationChannels->addChannel(Operation1Notify);
  NotificationChannels->addChannel(Operation2Notify);
}

NotificationObservable::~NotificationObservable()
{
  delete NotificationChannels;
}

void NotificationObservable::operation1(){
  OBSERVE;
  NOTIFY(Operation1Notify);
}

void NotificationObservable::operation2(){
  OBSERVE;
  NOTIFY(Operation2Notify);
}

/*********** NotificationObserver **************/

NotificationObserver::NotificationObserver(Notification_ptr notification) :
  Observer("NotificationObserver"),
  requestedNotification(notification),
  wasNotified(false)
{}

NotificationObserver::~NotificationObserver()
{}

void NotificationObserver::update(Observable*){}
void NotificationObserver::subjectKilled(Observable*){}
void NotificationObserver::recieveNotification(Observable *publisher, Notification_ptr notification){
  ASSERT(requestedNotification==notification,"Notification received that was not requested");
  wasNotified = true;
}

/**************** ObservableSet ****************/

ObservableSet::ObservableSet(int _num) :
  Observable("ObservableCollection"),
  num(_num)
{
  for(int i=0; i<num; ++i){
    SimpleObservable *content = new SimpleObservable();
    content->signOn(this);
    theSet.insert(content);
  }
}

ObservableSet::~ObservableSet(){
  set::iterator iter;
  for(iter=theSet.begin(); iter!=theSet.end(); ++iter ){
    delete (*iter);
  }
}

ObservableSet::iterator ObservableSet::begin(){
  return iterator(theSet.begin(),this);
}

ObservableSet::iterator ObservableSet::end(){
  return iterator(theSet.end(),this);
}

/************** ObservableMap ******************/

ObservableMap::ObservableMap(int _num) :
  Observable("ObservableCollection"),
  num(_num)
{
  for(int i=0; i<num; ++i){
    SimpleObservable *content = new SimpleObservable();
    content->signOn(this);
    theSet.insert(std::make_pair(i,content));
  }
}

ObservableMap::~ObservableMap(){
  set::iterator iter;
  for(iter=theSet.begin(); iter!=theSet.end(); ++iter ){
    delete iter->second;
  }
}

ObservableMap::iterator ObservableMap::begin(){
  return iterator(theSet.begin(),this);
}

ObservableMap::iterator ObservableMap::end(){
  return iterator(theSet.end(),this);
}

