/*
 * Cacheable.hpp
 *
 *  Created on: Feb 2, 2010
 *      Author: crueger
 */

#ifndef CACHEABLE_HPP_
#define CACHEABLE_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Observer/Observable.hpp"
#include "Observer/Observer.hpp"
#include <boost/function.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/thread/locks.hpp>
#include <boost/thread/recursive_mutex.hpp>

#include "CodePatterns/Assert.hpp"

#include "CodePatterns/Observer/Notification.hpp"

#ifndef NO_CACHING

  template<typename T>
  class Cacheable : public Observer
  {
    // we define the states of the cacheable so we can do very fast state-checks
    class State{
    public:
      State(Cacheable * const _owner) :
        busy(false),
        owner(_owner)
        {}
      virtual T& getValue()=0;
      virtual void invalidate()=0;
      virtual bool isValid()=0;
      virtual void enter()=0;
      bool isBusy(){
        return busy;
      }
    virtual std::string getName()=0;
    protected:
      bool busy;
      Cacheable * const owner;
    };

    class InvalidState : public State{
    public:
      InvalidState(Cacheable * const _owner):
        State(_owner)
        {}

      virtual T& getValue(){
        // set the state to valid
        State::owner->switchState(State::owner->validState);
        // get the value from the now valid state
        return State::owner->state->getValue();
      }

      virtual void invalidate(){
        // nothing to do on this message
      }

      virtual bool isValid(){
        return false;
      }

      virtual void enter(){
        // nothing to do when entering this
      }

      virtual std::string getName(){
        return "invalid";
      }
    };

    class ValidState : public State{
    public:
      ValidState(Cacheable * const _owner) :
        State(_owner)
        {}

      virtual T& getValue(){
        return content;
      }

      virtual void invalidate(){
        State::owner->switchState(State::owner->invalidState);
      }

      virtual bool isValid(){
        return true;
      }

      virtual void enter(){
        State::busy= true;
        // as soon as we enter the valid state we recalculate
        content = State::owner->recalcMethod();
        State::busy = false;
      }

      virtual std::string getName(){
        return "valid";
      }
    private:
      T content;
    };

    class DestroyedState : public State {
    public:
      DestroyedState(Cacheable * const _owner) :
        State(_owner)
        {}

      virtual T& getValue(){
        ASSERT(0,"Cannot get a value from a Cacheable after it's Observable has died");
        // we have to return a grossly invalid reference, because no value can be produced anymore
        return *(static_cast<T*>(0));
      }

      virtual void invalidate(){
        ASSERT(0,"Cannot invalidate a Cacheable after it's Observable has died");
      }

      virtual bool isValid(){
        ASSERT(0,"Cannot check validity of a Cacheable after it's Observable has died");
        return false;
      }

      virtual void enter(){
        // nothing to do when entering this state
      }

      virtual std::string getName(){
        return "destroyed";
      }
    };


  typedef boost::shared_ptr<State> state_ptr;

  public:
    Cacheable(
        const Observable * const _owner,
        const boost::function<T()> &_recalcMethod,
        const std::string &name,
        const Observable::channels_t &_channels = Observable::channels_t());
    virtual ~Cacheable();

    const bool isValid() const;
    const T operator*() const;

    // methods implemented for base-class Observer
    void update(Observable *subject);
    void recieveNotification(Observable *publisher, Notification_ptr notification);
    void subjectKilled(Observable *subject);
  private:
    void switchState(state_ptr newState);

    mutable state_ptr state;
    // pre-defined state so we don't have to construct to much
    state_ptr invalidState;
    state_ptr validState;
    // destroyed state is not predefined, because we rarely enter that state and never leave

    //!> mutex to ensure access is only per-thread
    mutable boost::recursive_mutex ownerLock;

    const Observable * const owner;
    const boost::function<T()> recalcMethod;

    //!> contains list of possible channels to enlist, if empty we signOn globally
    const Observable::channels_t channels;

    // de-activated copy constructor
    Cacheable(const Cacheable&);
  };


  template<typename T>
  Cacheable<T>::Cacheable(
      const Observable * const _owner,
      const boost::function<T()> &_recalcMethod,
      const std::string &_name,
      const Observable::channels_t &_channels) :
    Observer(_name + "(Cached)"),
    owner(_owner),
    recalcMethod(_recalcMethod),
    channels(_channels)
  {
    // create all states needed for this object
    invalidState = state_ptr(new InvalidState(this));
    validState = state_ptr(new ValidState(this));
    state = invalidState;
    // we sign on with the best(=lowest) priority, so cached values are recalculated before
    // anybody else might ask for updated values
    boost::lock_guard<boost::recursive_mutex> guard(ownerLock);
    if (owner != NULL) {
      if (channels.empty()) {
        owner->signOn(this,GlobalObservableInfo::PriorityLevel(int(-20)));
      } else {
        for (Observable::channels_t::const_iterator iter = channels.begin();
            iter != channels.end(); ++iter)
          owner->signOn(this,*iter,GlobalObservableInfo::PriorityLevel(int(-20)));
      }
    }
  }

  // de-activated copy constructor
  template<typename T>
  Cacheable<T>::Cacheable(const Cacheable&){
    ASSERT(0,"Cacheables should never be copied");
  }

  template<typename T>
  const T Cacheable<T>::operator*() const{
    // we can only use the cacheable when the owner is not changing at the moment
    boost::lock_guard<boost::recursive_mutex> guard(ownerLock);
    if ((owner == NULL) || (!owner->isBlocked())) {
      return state->getValue();
    }
    else{
      return recalcMethod();
    }
  }

  template<typename T>
  Cacheable<T>::~Cacheable()
  {
    boost::lock_guard<boost::recursive_mutex> guard(ownerLock);
    if (owner != NULL) {
      if (channels.empty()) {
        owner->signOff(this);
      } else {
        for (Observable::channels_t::const_iterator iter = channels.begin();
            iter != channels.end(); ++iter)
          owner->signOff(this,*iter);
      }
      const_cast<const Observable *&>(owner) = NULL;
    }
  }

  template<typename T>
  const bool Cacheable<T>::isValid() const{
    return state->isValid();
  }

  template<typename T>
  void Cacheable<T>::update(Observable *subject) {
    ASSERT( channels.empty(),
        "Cacheable<T>::update() - we are listening only to global updates.");
    state->invalidate();
  }

  template<typename T>
  void Cacheable<T>::recieveNotification(Observable *publisher, Notification_ptr notification) {
    boost::lock_guard<boost::recursive_mutex> guard(ownerLock);
    if (publisher == owner) {
      ASSERT( !channels.empty(),
          "Cacheable<T>::update() - we are not listening to global updates.");
      const Observable::channels_t::const_iterator iter = std::find(
          channels.begin(), channels.end(), notification->getChannelNo());
      if (iter != channels.end())
        state->invalidate();
    }
  }

  template<typename T>
  void Cacheable<T>::subjectKilled(Observable *subject) {
    state_ptr destroyed = state_ptr(new DestroyedState(this));
    switchState(destroyed);
    boost::lock_guard<boost::recursive_mutex> guard(ownerLock);
    const_cast<const Observable *&>(owner) = NULL;
  }

  template<typename T>
  void Cacheable<T>::switchState(state_ptr newState){
    ASSERT(!state->isBusy(),"LOOP DETECTED: Cacheable state switched while recalculating.\nDid the recalculation trigger the Observable?");
#ifdef LOG_OBSERVER
    observerLog().addMessage() << "## Cacheable " << observerLog().getName(this) << " changed state (" << state->getName()
                               << "->" << newState->getName() << ")";
#endif
    state = newState;
    state->enter();
  }

#else
  template<typename T>
  class Cacheable : public Observer
  {
  public:
    Cacheable(
        const Observable * const _owner,
        boost::function<T()> &_recalcMethod,
        std::string name,
        const Observable::channels_t &_channels);
    virtual ~Cacheable();

    const bool isValid() const;
    const T operator*() const;

    // methods implemented for base-class Observer
    void update(Observable *subject);
    void recieveNotification(Observable *publisher, Notification_ptr notification);
    void subjectKilled(Observable *subject);
  private:

    boost::function<T()> recalcMethod;
  };

  template<typename T>
  Cacheable<T>::Cacheable(
      const Observable * const _owner,
      boost::function<T()> &_recalcMethod,
      std::string name,
      const Observable::channels_t &_channels) :
    Observer(name),
    recalcMethod(_recalcMethod)
  {}

  template<typename T>
  const T Cacheable<T>::operator*() const{
    return recalcMethod();
  }

  template<typename T>
  Cacheable<T>::~Cacheable()
  {}

  template<typename T>
  const bool Cacheable<T>::isValid() const{
    return true;
  }

  template<typename T>
  void Cacheable<T>::update(Observable *subject) {
    ASSERT(0, "Cacheable::update should never be called when caching is disabled");
  }

  template<typename T>
  void Cacheable<T>::recieveNotification(Observable *publisher, Notification_ptr notification) {
    ASSERT(0, "Cacheable::recieveNotification should never be called when caching is disabled");
  }

  template<typename T>
  void Cacheable<T>::subjectKilled(Observable *subject){
    ASSERT(0, "Cacheable::subjectKilled should never be called when caching is disabled");
  }
#endif

#endif /* CACHEABLE_HPP_ */
