/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2011 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * FragmentController.cpp
 *
 *  Created on: Nov 27, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// boost asio needs specific operator new
#include <boost/asio.hpp>

#include "CodePatterns/MemDebug.hpp"

#include "Connection.hpp" // Must come before boost/serialization headers.
#include "CodePatterns/Info.hpp"
#include "CodePatterns/Log.hpp"

#include "Controller/FragmentController.hpp"
#include "Controller/Commands/CheckResultsOperation.hpp"
#include "Controller/Commands/GetNextJobIdOperation.hpp"
#include "Controller/Commands/ReceiveResultsOperation.hpp"
#include "Controller/Commands/SendJobsOperation.hpp"
#include "Controller/Commands/ShutdownOperation.hpp"

/** Constructor of class FragmentController.
 *
 * \param io_service io_service for the asynchronous operations
 * \param _host hostname of server that accepts jobs
 * \param _service of server
 */
FragmentController::FragmentController(
    boost::asio::io_service& io_service) :
    connection_(io_service)
{
  Info info(__FUNCTION__);

  // insert commands into registry
  Commands.registerInstance(new CheckResultsOperation(connection_));
  Commands.registerInstance(new GetNextJobIdOperation(connection_));
  Commands.registerInstance(new ReceiveResultsOperation(connection_));
  Commands.registerInstance(new SendJobsOperation(connection_));
  Commands.registerInstance(new ShutdownOperation(connection_));
}

/** Destructor of class FragmentController.
 *
 */
FragmentController::~FragmentController()
{
  Commands.cleanup();
}

/** Getter for Exitflag of all any Operation.
 *
 * We simply go through all of them and check which failed.
 *
 * @return Combined Exitflag of all Operations in \a Commands.
 */
size_t FragmentController::getExitflag() const
{
  for (CommandRegistry::const_iterator iter = Commands.getBeginIter();
      iter != Commands.getEndIter(); ++iter)
    if ((iter->second)->getExitflag() != 0)
      return (iter->second)->getExitflag();
  return 0;
}

