/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2011-2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * ReceiveResultsOperation.cpp
 *
 *  Created on: Dec 11, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// boost asio needs specific operator new
#include <boost/asio.hpp>

#include "CodePatterns/MemDebug.hpp"

#include <boost/bind.hpp>
#include <iostream>
#include <vector>
#include "Connection.hpp" // Must come before boost/serialization headers.
#include <boost/serialization/vector.hpp>
#include "CodePatterns/Info.hpp"
#include "CodePatterns/Log.hpp"
#include "ControllerChoices.hpp"
#include "Results/FragmentResult.hpp"

#include "Controller/Commands/ReceiveResultsOperation.hpp"

/** Handle connect operation to send results.
 *
 * \param e error code if something went wrong
 * \param endpoint_iterator endpoint of the connection
 */
void ReceiveResultsOperation::handle_connect(const boost::system::error_code& e,
    boost::asio::ip::tcp::resolver::iterator endpoint_iterator)
{
  Info info(__FUNCTION__);
  if (!e)
  {
    // Successfully established connection. Give choice.
    enum ControllerChoices choice = ReceiveResults;
    connection_.async_write(choice,
      boost::bind(&ReceiveResultsOperation::handle_ReceivingResults, this,
      boost::asio::placeholders::error));
  } else if (endpoint_iterator != boost::asio::ip::tcp::resolver::iterator()) {
    // Try the next endpoint.
    connection_.socket().close();
    boost::asio::ip::tcp::endpoint endpoint = *endpoint_iterator;
    connection_.socket().async_connect(endpoint,
      boost::bind(&ReceiveResultsOperation::handle_connect, this,
      boost::asio::placeholders::error, ++endpoint_iterator));
  } else {
    // An error occurred. Log it and return. Since we are not starting a new
    // operation the io_service will run out of work to do and the client will
    // exit.
    Exitflag = ErrorFlag;
    ELOG(1, e.message());
  }
}

/** Callback function when preparing to receive results
 *
 * \param e error code if something went wrong
 */
void ReceiveResultsOperation::handle_ReceivingResults(const boost::system::error_code& e)
{
  Info info(__FUNCTION__);
  if (!e)
  {
    // The connection::async_read() function will automatically
    // decode the data that is written to the underlying socket.
    LOG(1, "INFO: Sending "+toString(results.size())+" results ...");
    connection_.async_read(results,
      boost::bind(&ReceiveResultsOperation::handle_ReceivedResults, this,
      boost::asio::placeholders::error));
  }
  else
  {
    // An error occurred.
    Exitflag = ErrorFlag;
    ELOG(1, e.message());
  }
}

/** Callback function when results have been received.
 *
 * \param e error code if something went wrong
 */
void ReceiveResultsOperation::handle_ReceivedResults(const boost::system::error_code& e)
{
  Info info(__FUNCTION__);

  LOG(1, "INFO: Received "+toString(results.size())+" results.");

  ReceiveResultsOperation::handle_FinishOperation(e);
}

/** Getter for results.
 *
 * \sa calculateResults()
 * \return vector of results for the added jobs (\sa addJobs()).
 */
std::vector<FragmentResult::ptr> ReceiveResultsOperation::getResults()
{
  Info info(__FUNCTION__);
  return results;
}
