/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * Shape.cpp
 *
 *  Created on: Jun 18, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Helpers/MemDebug.hpp"

#include "Shape.hpp"
#include "Shape_impl.hpp"


#include "Helpers/Assert.hpp"
#include "LinearAlgebra/Vector.hpp"

Shape::Shape(const Shape& src) :
  impl(src.getImpl())
{}

Shape::~Shape(){}

bool Shape::isInside(const Vector &point) const{
  return impl->isInside(point);
}

std::vector<Vector> Shape::getHomogeneousPointsOnSurface(const size_t N) const {
  return impl->getHomogeneousPointsOnSurface(N);
}

Shape::Shape(Shape::impl_ptr _impl) :
    impl(_impl)
{}

Shape &Shape::operator=(const Shape& rhs){
  if(&rhs!=this){
    impl=rhs.getImpl();
  }
  return *this;
}

Shape::impl_ptr Shape::getImpl() const{
  return impl;
}

// allows arbitrary friendship, but only if implementation is known
Shape::impl_ptr getShapeImpl(const Shape &shape){
  return shape.getImpl();
}

/***************************** Some simple Shapes ***************************/

Shape Everywhere(){
  static Shape::impl_ptr impl = Shape::impl_ptr(new Everywhere_impl());
  return Shape(impl);
}

Shape Nowhere(){
  static Shape::impl_ptr impl = Shape::impl_ptr(new Nowhere_impl());
  return Shape(impl);
}

/****************************** Operators ***********************************/

// AND

AndShape_impl::AndShape_impl(const Shape::impl_ptr &_lhs, const Shape::impl_ptr &_rhs) :
  lhs(_lhs),rhs(_rhs)
{}

AndShape_impl::~AndShape_impl(){}

bool AndShape_impl::isInside(const Vector &point){
  return lhs->isInside(point) && rhs->isInside(point);
}

std::vector<Vector> AndShape_impl::getHomogeneousPointsOnSurface(const size_t N) const {
  std::vector<Vector> PointsOnSurface_lhs = lhs->getHomogeneousPointsOnSurface(N);
  std::vector<Vector> PointsOnSurface_rhs = rhs->getHomogeneousPointsOnSurface(N);
  std::vector<Vector> PointsOnSurface;

  for (std::vector<Vector>::const_iterator iter = PointsOnSurface_lhs.begin(); iter != PointsOnSurface_lhs.end(); ++iter) {
    if (rhs->isInside(*iter))
      PointsOnSurface.push_back(*iter);
  }
  for (std::vector<Vector>::const_iterator iter = PointsOnSurface_rhs.begin(); iter != PointsOnSurface_rhs.end(); ++iter) {
    if (lhs->isInside(*iter))
      PointsOnSurface.push_back(*iter);
  }

  return PointsOnSurface;
}


Shape operator&&(const Shape &lhs,const Shape &rhs){
  Shape::impl_ptr newImpl = Shape::impl_ptr(new AndShape_impl(getShapeImpl(lhs),getShapeImpl(rhs)));
  return Shape(newImpl);
}

// OR

OrShape_impl::OrShape_impl(const Shape::impl_ptr &_lhs, const Shape::impl_ptr &_rhs) :
  lhs(_lhs),rhs(_rhs)
{}

OrShape_impl::~OrShape_impl(){}

bool OrShape_impl::isInside(const Vector &point){
  return rhs->isInside(point) || lhs->isInside(point);
}

std::vector<Vector> OrShape_impl::getHomogeneousPointsOnSurface(const size_t N) const {
  std::vector<Vector> PointsOnSurface_lhs = lhs->getHomogeneousPointsOnSurface(N);
  std::vector<Vector> PointsOnSurface_rhs = rhs->getHomogeneousPointsOnSurface(N);
  std::vector<Vector> PointsOnSurface;

  for (std::vector<Vector>::const_iterator iter = PointsOnSurface_lhs.begin(); iter != PointsOnSurface_lhs.end(); ++iter) {
    if (!rhs->isInside(*iter))
      PointsOnSurface.push_back(*iter);
  }
  for (std::vector<Vector>::const_iterator iter = PointsOnSurface_rhs.begin(); iter != PointsOnSurface_rhs.end(); ++iter) {
    if (!lhs->isInside(*iter))
      PointsOnSurface.push_back(*iter);
  }

  return PointsOnSurface;
}

Shape operator||(const Shape &lhs,const Shape &rhs){
  Shape::impl_ptr newImpl = Shape::impl_ptr(new OrShape_impl(getShapeImpl(lhs),getShapeImpl(rhs)));
  return Shape(newImpl);
}

// NOT

NotShape_impl::NotShape_impl(const Shape::impl_ptr &_arg) :
  arg(_arg)
{}

NotShape_impl::~NotShape_impl(){}

bool NotShape_impl::isInside(const Vector &point){
  return !arg->isInside(point);
}

std::vector<Vector> NotShape_impl::getHomogeneousPointsOnSurface(const size_t N) const {
  // surfaces are the same, only normal direction is different
  return arg->getHomogeneousPointsOnSurface(N);
}

Shape operator!(const Shape &arg){
  Shape::impl_ptr newImpl = Shape::impl_ptr(new NotShape_impl(getShapeImpl(arg)));
  return Shape(newImpl);
}
