/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * GLMoleculeObject_atom.cpp
 *
 *  Created on: Aug 17, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "GLMoleculeObject_atom.hpp"

#include <Qt3D/qglbuilder.h>
#include <Qt3D/qglscenenode.h>
#include <Qt3D/qglsphere.h>

#include "CodePatterns/MemDebug.hpp"

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"

#include "Descriptors/AtomIdDescriptor.hpp"
#include "element.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "World.hpp"

static QGLSceneNode * createAtom(QObject *parent)
{
  QGLBuilder builder;
  builder << QGLSphere(0.5);
  QGLSceneNode *n = builder.finalizedSceneNode();
  n->setParent(parent);
  return n;
}

GLMoleculeObject_atom::GLMoleculeObject_atom(QObject *parent, const atom *atomref) :
  GLMoleculeObject(createAtom(parent), parent), Observer("GLMoleculeObject_atom"), _atom(atomref)
{
  // sign on as observer (obtain non-const instance before)
  atomref->signOn(this, atomref->getChannel(AtomObservable::IndexChanged));
  atomref->signOn(this, atomref->getChannel(AtomObservable::PositionChanged));
  atomref->signOn(this, atomref->getChannel(AtomObservable::ElementChanged));
  atomref->signOn(this, atomref->getChannel(AtomObservable::BondsChanged));

  // set the object's id
  resetProperties();

  std::cout << "Created sphere for atom " << _atom->getId() << "." << endl;

  connect( this, SIGNAL(clicked()), this, SLOT(wasClicked()));
}

GLMoleculeObject_atom::~GLMoleculeObject_atom()
{
  _atom->signOff(this, _atom->getChannel(AtomObservable::IndexChanged));
  _atom->signOff(this, _atom->getChannel(AtomObservable::PositionChanged));
  _atom->signOff(this, _atom->getChannel(AtomObservable::ElementChanged));
  _atom->signOff(this, _atom->getChannel(AtomObservable::BondsChanged));
}

void GLMoleculeObject_atom::update(Observable *publisher)
{
  LOG(0, "GLMoleculeObject_atom::update() - called fro atom "+toString(_atom->getId())+".");
  resetProperties();
}

void GLMoleculeObject_atom::resetPosition()
{
  const Vector Position = _atom->getPosition();
  LOG(1, "GLMoleculeObject_atom::resetIndex() - new position is "+toString(Position)+".");
  setPosition(QVector3D(Position[0], Position[1], Position[2]));
}

void GLMoleculeObject_atom::resetElement()
{
  size_t elementno = 0;
  if (_atom->getType() != NULL) {
    elementno = _atom->getType()->getNumber();
  } else { // if no element yet, set to hydrogen
    elementno = 1;
  }
  LOG(1, "GLMoleculeObject_atom::resetIndex() - new element number is "+toString(elementno)+".");

  // set materials
  QGLMaterial *elementmaterial = getMaterial(elementno);
  ASSERT(elementmaterial != NULL,
      "GLMoleculeObject_atom::GLMoleculeObject_atom() - QGLMaterial ref from getter function is NULL.");
  setMaterial(elementmaterial);
  QGLMaterial *hovermaterial = getMaterial(0); // 0 is the hover material
  ASSERT(hovermaterial != NULL,
      "GLMoleculeObject_atom::GLMoleculeObject_atom() - QGLMaterial ref from getter function for hover is NULL.");
  setHoverMaterial(hovermaterial);

  // set scale
  double radius = 0.;
  if (_atom->getType() != NULL) {
    radius = _atom->getType()->getVanDerWaalsRadius();
  } else {
    radius = 0.5;
  }
  setScale( radius );
}

void GLMoleculeObject_atom::resetIndex()
{
  const size_t atomno = _atom->getId();
  LOG(1, "GLMoleculeObject_atom::resetIndex() - new index is "+toString(atomno)+".");
  setObjectId(atomno);
}

void GLMoleculeObject_atom::resetProperties()
{
  // set position
  resetPosition();

  // set element
  resetElement();

  // set the object's id
  resetIndex();
}

void GLMoleculeObject_atom::subjectKilled(Observable *publisher)
{}

void GLMoleculeObject_atom::recieveNotification(Observable *publisher, Notification_ptr notification)
{
  LOG(0, "GLMoleculeObject_atom::recieveNotification() - atom "
      +toString(_atom->getId())+" got notification type "
      +toString(notification->getChannelNo())+".");
  switch (notification->getChannelNo()) {
    case AtomObservable::ElementChanged:
      resetElement();
      break;
    case AtomObservable::IndexChanged:
      resetIndex();
      break;
    case AtomObservable::PositionChanged:
      resetPosition();
      break;
    case AtomObservable::BondsChanged:
      emit BondsChanged(_atom);
      break;
    default:
      //setProperties();
      break;
  }
}

void GLMoleculeObject_atom::wasClicked()
{
  qDebug("GLMoleculeObject_atom: atom %d has been clicked", _atom->getId());
  emit clicked(_atom->getId());
}
