/** \file FormatParserStorage.hpp
 *
 *  date: Jun, 22 2010
 *  author: heber
 *
 */

#ifndef FORMATPARSERSTORAGE_HPP_
#define FORMATPARSERSTORAGE_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/Singleton.hpp"

#include <string>
#include <map>
#include <vector>

#include "Parser/ParserTypes.hpp"
#include "Parser/FormatParser.hpp"
#include "Parser/MpqcParser.hpp"
#include "Parser/PcpParser.hpp"
#include "Parser/PdbParser.hpp"
#include "Parser/TremoloParser.hpp"
#include "Parser/XyzParser.hpp"

class atom;

template <class T> enum ParserTypes getPType();
template <> enum ParserTypes getPType<MpqcParser>();
template <> enum ParserTypes getPType<PcpParser>();
template <> enum ParserTypes getPType<PdbParser>();
template <> enum ParserTypes getPType<TremoloParser>();
template <> enum ParserTypes getPType<XyzParser>();


class FormatParserStorage : public Singleton<FormatParserStorage> {
  friend class Singleton<FormatParserStorage>;
public:

  bool add(std::string type);
  bool add(ParserTypes type);

  bool load(std::istream &input, std::string suffix);
  bool save(std::ostream &output, std::string suffix, const std::vector<atom *> &atoms);
  bool saveSelectedAtoms(std::ostream &output, std::string suffix);
  bool saveSelectedMolecules(std::ostream &output, std::string suffix);
  bool saveWorld(std::ostream &output, std::string suffix);
  
  FormatParser &get(enum ParserTypes _type);

  ParserTypes getTypeFromName(std::string type);
  ParserTypes getTypeFromSuffix(std::string type);

  void SetOutputPrefixForAll(std::string &_prefix);
  void SaveAll();

private:
  // private constructors as this is a singleton
  FormatParserStorage();
  ~FormatParserStorage();
  
  // list of allocated parsers
  std::vector<FormatParser *> ParserList;

  // list of allocated strams
  std::vector<std::ofstream *> ParserStream;

  // which parser is already present
  std::vector<bool> ParserPresent;

  // default suffix of each parser type
  std::map<ParserTypes, std::string> ParserSuffixes;
  std::map<std::string, ParserTypes> ParserLookupSuffixes;

  // function pointers to each add...()
  std::map< ParserTypes, void (FormatParserStorage::*)() > ParserAddFunction;

  // type name of each parser type and reverse lookup
  std::map<ParserTypes, std::string> ParserNames;
  std::map<std::string, ParserTypes> ParserLookupNames;


  // prefix of the filenames to use on save
  std::string prefix;

public:

  template<class ParserT> void addParser()
  {
    enum ParserTypes Ptype = getPType<ParserT>();
    if (!ParserPresent[Ptype]) {
      ParserList[Ptype] = new ParserT();
      ParserPresent[Ptype] = true;
    } else {
      ASSERT(ParserNames.find(Ptype) != ParserNames.end(),
          "FormatParserStorage::addParser() - ParserNames unknown for type"+toString((size_t)Ptype)+".");
      ASSERT(ParserSuffixes.find(Ptype) != ParserSuffixes.end(),
          "FormatParserStorage::addParser() - ParserSuffixes unknown for type"+toString((size_t)Ptype)+".");
      LOG(2, "INFO: Parser " << ParserNames[Ptype] << " is already present." << std::endl
          << "Note that you don't need to add '-o " 
          << ParserSuffixes[Ptype] << "' if the input file is non-empty and of type " 
          << ParserSuffixes[Ptype] << "." << std::endl);
    }
  }

  template<class ParserT> ParserT &getParser() 
  {
    enum ParserTypes Ptype = getPType<ParserT>();
    if(!ParserPresent[Ptype])
      addParser<ParserT>();
    return dynamic_cast<ParserT &>(*ParserList[Ptype]);
  }

};

#endif // FORMATPARSERSTORAGE_HPP_
