/*
 * TextMenu.cpp
 *
 *  Created on: Dec 10, 2009
 *      Author: crueger
 */

#include "Helpers/MemDebug.hpp"

#include <boost/bind.hpp>
#include <iostream>
#include <cmath>
#include "Menu/TextMenu.hpp"
#include "Menu/MenuItem.hpp"
#include "Helpers/Assert.hpp"


/**
 * produce a text menu with a given title.
 * The text will later be displayed using the stream passed to the constructor.
 */
TextMenu::TextMenu(ostream& _outputter, string _title, char _spacer,int _length) :
  defaultItem(0),
  outputter(_outputter),
  title(_title),
  spacer(_spacer),
  length(_length),
  quit(false)
{
}

TextMenu::~TextMenu()
{
  for(list<MenuItem*>::iterator it=items.begin(); it != items.end(); it++)
    delete (*it);
}


void TextMenu::addItem(MenuItem* item) {
  items.push_back(item);
}

void TextMenu::removeItem(MenuItem* item) {
  items.remove(item);
}

void TextMenu::doQuit(){
  quit = true;
}

bool TextMenu::hasQuit(){
  return quit;
}

void TextMenu::showEntry(MenuItem* entry){
  if(entry->isActive()==false){
    outputter << "(";
  }
  outputter << entry->formatEntry();
  if(entry->isActive()==false){
    outputter << ")";
  }
  outputter << "\n";
}

void TextMenu::display() {
  char choice;
  bool somethingChosen = false;
  quit = false;
  do {
    int pre = floor((length - title.length()) /2.0);
    int post = ceil((length - title.length()) /2.0);
    for(int i=0;i<pre;i++)
      outputter << spacer;
    outputter << title;
    for(int i=0;i<post;i++)
          outputter << spacer;
    outputter << '\n';
    for_each(items.begin(), items.end(), boost::bind(&TextMenu::showEntry,this,_1));
    outputter.flush();

    cin >> choice;

    list<MenuItem*>::iterator iter;
    for(iter = items.begin(); iter!=items.end();iter++){
      if((*iter)->isActive()){
        somethingChosen |= (*iter)->checkTrigger(choice);
      }
    }
    // see if something was chosen and call default Item if not
    if(!somethingChosen) {
      if(defaultItem){
        defaultItem->doTrigger();
      }
      else{
        outputter << "Invalid Choice!" <<  endl;
      }
    }
  }while (!hasQuit());
}

string TextMenu::getTitle(){
  return title;
}

void TextMenu::addDefault(MenuItem* _defaultItem) {
  defaultItem = _defaultItem;
}

/****************************** Contained Actions ****************/

const string TextMenu::LeaveAction::nameBase = "Leave menu: ";

TextMenu::LeaveAction::LeaveAction(TextMenu* _menu) :
  Action(nameBase+_menu->getTitle()),
  menu(_menu)
{}

TextMenu::LeaveAction::~LeaveAction(){}

bool TextMenu::LeaveAction::canUndo(){
  return false;
}

bool TextMenu::LeaveAction::shouldUndo(){
  return false;
}

Dialog* TextMenu::LeaveAction::fillDialog(Dialog *dialog){
  ASSERT(dialog,"No Dialog given when filling action dialog");
  return dialog;
}


Action::state_ptr TextMenu::LeaveAction::performCall(){
  menu->doQuit();
  return Action::success;
}


Action::state_ptr TextMenu::LeaveAction::performUndo(Action::state_ptr){
  ASSERT(0,"Cannot undo leaving a menu");
  return Action::success;
}

Action::state_ptr TextMenu::LeaveAction::performRedo(Action::state_ptr){
  ASSERT(0,"Cannot redo leaving a menu");
  return Action::success;
}
