/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * QtDialog.cpp
 *
 *  Created on: Jan 18, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "UIElements/Qt4/QtDialog.hpp"

#include <boost/lexical_cast.hpp>

#include <string>
#include <sstream>
#include <limits>

#include <QtGui/QDoubleSpinBox>
#include <Qt/qboxlayout.h>
#include <Qt/qcombobox.h>
#include <Qt/qdialogbuttonbox.h>
#include <Qt/qlabel.h>
#include <Qt/qlineedit.h>
#include <Qt/qlistwidget.h>
#include <Qt/qpushbutton.h>
#include <Qt/qspinbox.h>
#include <Qt/qtablewidget.h>

#include <boost/lexical_cast.hpp>

#include "Helpers/MemDebug.hpp"

#include "World.hpp"
#include "periodentafel.hpp"
#include "atom.hpp"
#include "element.hpp"
#include "molecule.hpp"
#include "Descriptors/AtomIdDescriptor.hpp"
#include "Descriptors/MoleculeIdDescriptor.hpp"
#include "LinearAlgebra/Matrix.hpp"
#include "Box.hpp"


using namespace std;

QtDialog::QtDialog() :
    QDialog(0)
{
  // creating and filling of the Dialog window
  mainLayout = new QVBoxLayout();
  inputLayout = new QVBoxLayout();
  buttonLayout = new QVBoxLayout();
  setLayout(mainLayout);
  mainLayout->addLayout(inputLayout);
  mainLayout->addLayout(buttonLayout);
  buttons = new QDialogButtonBox(QDialogButtonBox::Ok| QDialogButtonBox::Cancel);
  buttonLayout->addWidget(buttons);

  // Disable the ok button until something was entered
  buttons->button(QDialogButtonBox::Ok)->setEnabled(false);

  // connect the buttons to their appropriate slots
  connect(buttons, SIGNAL(accepted()), this, SLOT(accept()));
  connect(buttons, SIGNAL(rejected()), this, SLOT(reject()));
}

QtDialog::~QtDialog()
{
}

bool QtDialog::display(){
  // Button state might have changed by some update that
  // was done during query construction. To make sure
  // the state is correct, we just call update one more time.
  update();
  if(exec()) {
    setAll();
    return true;
  }
  else {
    return false;
  }
}

void QtDialog::update(){
  buttons->button(QDialogButtonBox::Ok)->setEnabled(checkAll());
}

/************************** Query Infrastructure ************************/

void QtDialog::queryEmpty(const char* title, std::string)
{
  registerQuery(new EmptyQtQuery(title,inputLayout,this));
}

void QtDialog::queryBoolean(const char* title,string)
{
  registerQuery(new BooleanQtQuery(title,inputLayout,this));
}

void QtDialog::queryAtom(const char* title, std::string)
{
  registerQuery(new AtomQtQuery(title,inputLayout,this));
}

void QtDialog::queryAtoms(const char* title, std::string)
{
  registerQuery(new AtomsQtQuery(title,inputLayout,this));
}

void QtDialog::queryBox(const char* title, std::string)
{
  registerQuery(new BoxQtQuery(title,inputLayout,this));
}

void QtDialog::queryInt(const char *title,string)
{
  registerQuery(new IntQtQuery(title,inputLayout,this));
}

void QtDialog::queryInts(const char *title,string)
{
  registerQuery(new IntsQtQuery(title,inputLayout,this));
}

void QtDialog::queryDouble(const char* title,string)
{
  registerQuery(new DoubleQtQuery(title,inputLayout,this));
}

void QtDialog::queryDoubles(const char* title,string)
{
  registerQuery(new DoublesQtQuery(title,inputLayout,this));
}

void QtDialog::queryString(const char* title,string)
{
  registerQuery(new StringQtQuery(title,inputLayout,this));
}

void QtDialog::queryStrings(const char* title,string)
{
  registerQuery(new StringsQtQuery(title,inputLayout,this));
}

void QtDialog::queryMolecule(const char *title,string)
{
  registerQuery(new MoleculeQtQuery(title,inputLayout,this));
}

void QtDialog::queryMolecules(const char *title,string)
{
  registerQuery(new MoleculesQtQuery(title,inputLayout,this));
}

void QtDialog::queryVector(const char* title, bool check,string)
{
  registerQuery(new VectorQtQuery(title,check,inputLayout,this));
}

void QtDialog::queryVectors(const char* title, bool check,string)
{
  registerQuery(new VectorsQtQuery(title,check,inputLayout,this));
}

void QtDialog::queryElement(const char* title, std::string)
{
  registerQuery(new ElementQtQuery(title,inputLayout,this));
}

void QtDialog::queryElements(const char* title, std::string)
{
  registerQuery(new ElementsQtQuery(title,inputLayout,this));
}

void QtDialog::queryFile(const char* title, std::string)
{
  registerQuery(new FileQtQuery(title,inputLayout,this));
}

/************************** Query Objects *******************************/

QtDialog::BoxQtQuery::BoxQtQuery(string _title,QBoxLayout *_parent,QtDialog *_dialog) :
    Dialog::BoxQuery(_title),
    parent(_parent)
{
  thisLayout = new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));

  // init input table
  inputTable = new QTableWidget(3,3, _dialog);
  QStringList CoordinateList;
  CoordinateList << "x" << "y" << "z";
  inputTable->setHorizontalHeaderLabels(CoordinateList);
  inputTable->setVerticalHeaderLabels(CoordinateList);

  pipe = new BoxQtQueryPipe(tmp,_dialog, inputTable);

  // fill the box with current matrix
  const Matrix &domain = World::getInstance().getDomain().getM();
  for (int i=0;i<3;i++)
    for (int j=0;j<3;j++) {
      QTableWidgetItem *newItem = new QTableWidgetItem(tr("%1").arg(domain.at(i,j)));
      inputTable->setItem(i,j,newItem);
      pipe->update(i,j);
    }

  parent->addLayout(thisLayout);
  thisLayout->addWidget(titleLabel);
  thisLayout->addWidget(inputTable);

  connect(inputTable,SIGNAL(cellChanged(int,int)),pipe,SLOT(update(int,int)));
}

QtDialog::BoxQtQuery::~BoxQtQuery()
{
  delete pipe;
}

bool QtDialog::BoxQtQuery::handle() {
  return true;
}


QtDialog::AtomQtQuery::AtomQtQuery(string _title,QBoxLayout *_parent,QtDialog *_dialog) :
    Dialog::AtomQuery(_title),
    parent(_parent)
{
  thisLayout = new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));
  inputBox = new QComboBox();
  inputBox->insertItem(-1, QString("no atom"));
  std::vector<atom *> atoms = World::getInstance().getAllAtoms();
  for (std::vector<atom *>::const_iterator iter = atoms.begin(); iter != atoms.end(); ++iter)
    inputBox->insertItem((*iter)->getNr(),QString::fromStdString((*iter)->getName()));

  parent->addLayout(thisLayout);
  thisLayout->addWidget(titleLabel);
  thisLayout->addWidget(inputBox);

  pipe = new AtomQtQueryPipe(&tmp,_dialog, inputBox);
  pipe->update(inputBox->currentIndex());
  connect(inputBox,SIGNAL(currentIndexChanged(int)),pipe,SLOT(update(int)));
}

QtDialog::AtomQtQuery::~AtomQtQuery()
{
  delete pipe;
}

bool QtDialog::AtomQtQuery::handle() {
  return true;
}


QtDialog::AtomsQtQuery::AtomsQtQuery(string _title,QBoxLayout *_parent,QtDialog *_dialog) :
    Dialog::AtomsQuery(_title),
    parent(_parent)
{
  QHBoxLayout * thisHLayout = new QHBoxLayout();
  QVBoxLayout * thisV1Layout = new QVBoxLayout();
  QVBoxLayout * thisV2Layout = new QVBoxLayout();

  QLabel *titleLabel = new QLabel(QString(getTitle().c_str()));
  QLabel *inputLabel = new QLabel("Enter to add");
  QListWidget* inputList = new QListWidget();
  inputList->setSelectionMode(QAbstractItemView::ExtendedSelection);
  std::vector<atom *> atoms = World::getInstance().getAllAtoms();
  for (std::vector<atom *>::const_iterator iter = atoms.begin(); iter != atoms.end(); ++iter)
    inputBox->insertItem((*iter)->getNr(),QString((*iter)->getNr()));

  QLineEdit* inputBox = new QLineEdit();
  inputLabel->setBuddy(inputBox);
  titleLabel->setBuddy(inputList);
  QPushButton* AddButton = new QPushButton("Add");
  AddButton->setEnabled(false);
  QPushButton* RemoveButton = new QPushButton("Remove");
  RemoveButton->setEnabled(false);

  thisV1Layout->addWidget(titleLabel);
  thisV1Layout->addWidget(inputList);
  thisV2Layout->addWidget(inputLabel);
  thisV2Layout->addWidget(inputBox);
  thisV2Layout->addWidget(AddButton);
  thisV2Layout->addWidget(RemoveButton);
  parent->addLayout(thisHLayout);
  thisHLayout->addLayout(thisV1Layout);
  thisHLayout->addLayout(thisV2Layout);

  pipe = new AtomsQtQueryPipe(&tmp,_dialog,inputList);
  connect(inputList,SIGNAL(itemSelectionChanged()),pipe,SLOT(IntegerSelected()));
  connect(AddButton,SIGNAL(Clicked()),pipe,SLOT(add()));
  connect(RemoveButton,SIGNAL(Clicked()),pipe,SLOT(remove()));
}

QtDialog::AtomsQtQuery::~AtomsQtQuery()
{
  delete pipe;
}

bool QtDialog::AtomsQtQuery::handle() {
  return true;
}

QtDialog::IntQtQuery::IntQtQuery(string _title,QBoxLayout *_parent,QtDialog *_dialog) :
    Dialog::IntQuery(_title),
    parent(_parent)
{
  thisLayout = new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));
  inputBox = new QSpinBox();
  inputBox->setValue(0);
  parent->addLayout(thisLayout);
  thisLayout->addWidget(titleLabel);
  thisLayout->addWidget(inputBox);

  pipe = new IntQtQueryPipe(&tmp,_dialog);
  pipe->update(inputBox->value());
  connect(inputBox,SIGNAL(valueChanged(int)),pipe,SLOT(update(int)));
}

QtDialog::IntQtQuery::~IntQtQuery()
{
  delete pipe;
}

bool QtDialog::IntQtQuery::handle() {
  return true;
}


QtDialog::IntsQtQuery::IntsQtQuery(string _title,QBoxLayout *_parent,QtDialog *_dialog) :
    Dialog::IntsQuery(_title),
    parent(_parent)
{
  QHBoxLayout * thisHLayout = new QHBoxLayout();
  QVBoxLayout * thisV1Layout = new QVBoxLayout();
  QVBoxLayout * thisV2Layout = new QVBoxLayout();

  QLabel *titleLabel = new QLabel(QString(getTitle().c_str()));
  QLabel *inputLabel = new QLabel("Enter to add");
  QListWidget* inputList = new QListWidget();
  inputList->setSelectionMode(QAbstractItemView::ExtendedSelection);
  QLineEdit* inputBox = new QLineEdit();
  inputLabel->setBuddy(inputBox);
  titleLabel->setBuddy(inputList);
  QPushButton* AddButton = new QPushButton("Add");
  AddButton->setEnabled(false);
  QPushButton* RemoveButton = new QPushButton("Remove");
  RemoveButton->setEnabled(false);

  thisV1Layout->addWidget(titleLabel);
  thisV1Layout->addWidget(inputList);
  thisV2Layout->addWidget(inputLabel);
  thisV2Layout->addWidget(inputBox);
  thisV2Layout->addWidget(AddButton);
  thisV2Layout->addWidget(RemoveButton);
  parent->addLayout(thisHLayout);
  thisHLayout->addLayout(thisV1Layout);
  thisHLayout->addLayout(thisV2Layout);

  pipe = new QtQueryListPipe<int>(&tmp,_dialog,inputBox,inputList,AddButton,RemoveButton);
  connect(inputBox,SIGNAL(textChanged(const QString&)),pipe,SLOT(IntegerEntered(const QString&)));
  connect(inputList,SIGNAL(itemSelectionChanged()),pipe,SLOT(IntegerSelected()));
  connect(AddButton,SIGNAL(Clicked()),pipe,SLOT(AddValue()));
  connect(RemoveButton,SIGNAL(Clicked()),pipe,SLOT(RemoveRow()));
}

QtDialog::IntsQtQuery::~IntsQtQuery()
{
  delete pipe;
}

bool QtDialog::IntsQtQuery::handle() {
  return true;
}


QtDialog::DoubleQtQuery::DoubleQtQuery(string title,QBoxLayout *_parent,QtDialog *_dialog) :
    Dialog::DoubleQuery(title),
    parent(_parent)
{
  thisLayout = new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));
  inputBox = new QDoubleSpinBox();
  inputBox->setValue(0);
  inputBox->setRange(-numeric_limits<double>::max(),numeric_limits<double>::max());
  inputBox->setDecimals(3);
  parent->addLayout(thisLayout);
  thisLayout->addWidget(titleLabel);
  thisLayout->addWidget(inputBox);

  pipe = new DoubleQtQueryPipe(&tmp,_dialog);
  pipe->update(inputBox->value());
  connect(inputBox,SIGNAL(valueChanged(double)),pipe,SLOT(update(double)));
}

QtDialog::DoubleQtQuery::~DoubleQtQuery()
{
  delete pipe;
}

bool QtDialog::DoubleQtQuery::handle() {
  return true;
}


QtDialog::DoublesQtQuery::DoublesQtQuery(string title,QBoxLayout *_parent,QtDialog *_dialog) :
    Dialog::DoublesQuery(title),
    parent(_parent)
{
  QHBoxLayout * thisHLayout = new QHBoxLayout();
  QVBoxLayout * thisV1Layout = new QVBoxLayout();
  QVBoxLayout * thisV2Layout = new QVBoxLayout();

  QLabel *titleLabel = new QLabel(QString(getTitle().c_str()));
  QLabel *inputLabel = new QLabel("Enter to add");
  QListWidget* inputList = new QListWidget();
  inputList->setSelectionMode(QAbstractItemView::ExtendedSelection);
  QLineEdit* inputBox = new QLineEdit();
  inputLabel->setBuddy(inputBox);
  titleLabel->setBuddy(inputList);
  QPushButton* AddButton = new QPushButton("Add");
  AddButton->setEnabled(false);
  QPushButton* RemoveButton = new QPushButton("Remove");
  RemoveButton->setEnabled(false);

  thisV1Layout->addWidget(titleLabel);
  thisV1Layout->addWidget(inputList);
  thisV2Layout->addWidget(inputLabel);
  thisV2Layout->addWidget(inputBox);
  thisV2Layout->addWidget(AddButton);
  thisV2Layout->addWidget(RemoveButton);
  parent->addLayout(thisHLayout);
  thisHLayout->addLayout(thisV1Layout);
  thisHLayout->addLayout(thisV2Layout);

  pipe = new QtQueryListPipe<double>(&tmp,_dialog,inputBox,inputList,AddButton,RemoveButton);
  connect(inputBox,SIGNAL(textChanged(const QString&)),pipe,SLOT(IntegerEntered(const QString&)));
  connect(inputList,SIGNAL(itemSelectionChanged()),pipe,SLOT(IntegerSelected()));
  connect(AddButton,SIGNAL(Clicked()),pipe,SLOT(AddValue()));
  connect(RemoveButton,SIGNAL(Clicked()),pipe,SLOT(RemoveRow()));}

QtDialog::DoublesQtQuery::~DoublesQtQuery()
{
  delete pipe;
}

bool QtDialog::DoublesQtQuery::handle() {
  return true;
}


QtDialog::StringQtQuery::StringQtQuery(string _title,QBoxLayout *_parent,QtDialog *_dialog) :
    Dialog::StringQuery(_title),
    parent(_parent)
{
  thisLayout = new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));
  inputBox = new QLineEdit();
  parent->addLayout(thisLayout);
  thisLayout->addWidget(titleLabel);
  thisLayout->addWidget(inputBox);

  pipe = new StringQtQueryPipe(&tmp,_dialog);
  pipe->update(inputBox->text());
  connect(inputBox,SIGNAL(textChanged(const QString&)),pipe,SLOT(update(const QString&)));
}

QtDialog::StringQtQuery::~StringQtQuery()
{
  delete pipe;
}

// All values besides the empty std::string are valid
bool QtDialog::StringQtQuery::handle()
{
  return tmp!="";
}

QtDialog::StringsQtQuery::StringsQtQuery(string _title,QBoxLayout *_parent,QtDialog *_dialog) :
    Dialog::StringsQuery(_title),
    parent(_parent)
{
  QHBoxLayout * thisHLayout = new QHBoxLayout();
  QVBoxLayout * thisV1Layout = new QVBoxLayout();
  QVBoxLayout * thisV2Layout = new QVBoxLayout();

  QLabel *titleLabel = new QLabel(QString(getTitle().c_str()));
  QLabel *inputLabel = new QLabel("Enter to add");
  QListWidget* inputList = new QListWidget();
  inputList->setSelectionMode(QAbstractItemView::ExtendedSelection);
  QLineEdit* inputBox = new QLineEdit();
  inputLabel->setBuddy(inputBox);
  titleLabel->setBuddy(inputList);
  QPushButton* AddButton = new QPushButton("Add");
  AddButton->setEnabled(false);
  QPushButton* RemoveButton = new QPushButton("Remove");
  RemoveButton->setEnabled(false);

  thisV1Layout->addWidget(titleLabel);
  thisV1Layout->addWidget(inputList);
  thisV2Layout->addWidget(inputLabel);
  thisV2Layout->addWidget(inputBox);
  thisV2Layout->addWidget(AddButton);
  thisV2Layout->addWidget(RemoveButton);
  parent->addLayout(thisHLayout);
  thisHLayout->addLayout(thisV1Layout);
  thisHLayout->addLayout(thisV2Layout);

  pipe = new QtQueryListPipe<std::string>(&tmp,_dialog,inputBox,inputList,AddButton,RemoveButton);
  connect(inputBox,SIGNAL(textChanged(const QString&)),pipe,SLOT(IntegerEntered(const QString&)));
  connect(inputList,SIGNAL(itemSelectionChanged()),pipe,SLOT(IntegerSelected()));
  connect(AddButton,SIGNAL(Clicked()),pipe,SLOT(AddValue()));
  connect(RemoveButton,SIGNAL(Clicked()),pipe,SLOT(RemoveRow()));}

QtDialog::StringsQtQuery::~StringsQtQuery()
{
  delete pipe;
}

// All values besides the empty std::string are valid
bool QtDialog::StringsQtQuery::handle()
{
  // dissect by ","
  std::string::iterator olditer = temp.begin();
  for(string::iterator iter = temp.begin(); iter != temp.end(); ++iter) {
    if (*iter == ' ') {
      tmp.push_back(string(iter, olditer));
      olditer = iter;
    }
  }
  if (olditer != temp.begin())  // insert last part also
    tmp.push_back(string(olditer, temp.end()));

  return temp!="";
}

QtDialog::MoleculeQtQuery::MoleculeQtQuery(string _title, QBoxLayout *_parent,QtDialog *_dialog) :
    Dialog::MoleculeQuery(_title),
    parent(_parent)
{
  thisLayout = new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));
  inputBox = new QComboBox();
  // add all molecules to the combo box
  vector<molecule*> molecules = World::getInstance().getAllMolecules();
  for(vector<molecule*>::iterator iter  = molecules.begin();
      iter != molecules.end();
      ++iter) {
    std::stringstream sstr;
    sstr << (*iter)->IndexNr << "\t" << (*iter)->getName();
    inputBox->addItem(QString(sstr.str().c_str()),QVariant((*iter)->IndexNr));
  }
  parent->addLayout(thisLayout);
  thisLayout->addWidget(titleLabel);
  thisLayout->addWidget(inputBox);

  pipe = new MoleculeQtQueryPipe(&tmp,_dialog,inputBox);
  pipe->update(inputBox->currentIndex());
  connect(inputBox,SIGNAL(currentIndexChanged(int)),pipe,SLOT(update(int)));
}

QtDialog::MoleculeQtQuery::~MoleculeQtQuery()
{
  delete pipe;
}

// Handling is easy, since the GUI makes it impossible to select invalid values
bool QtDialog::MoleculeQtQuery::handle()
{
  return true;
}

QtDialog::MoleculesQtQuery::MoleculesQtQuery(string _title, QBoxLayout *_parent,QtDialog *_dialog) :
    Dialog::MoleculesQuery(_title),
    parent(_parent)
{
  thisLayout = new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));
  inputBox = new QComboBox();
  // add all molecules to the combo box
  vector<molecule*> molecules = World::getInstance().getAllMolecules();
  for(vector<molecule*>::iterator iter  = molecules.begin();
      iter != molecules.end();
      ++iter) {
    std::stringstream sstr;
    sstr << (*iter)->IndexNr << "\t" << (*iter)->getName();
    inputBox->addItem(QString(sstr.str().c_str()),QVariant((*iter)->IndexNr));
  }
  parent->addLayout(thisLayout);
  thisLayout->addWidget(titleLabel);
  thisLayout->addWidget(inputBox);

  pipe = new MoleculesQtQueryPipe(&tmp,_dialog,inputBox);
  pipe->update(inputBox->currentIndex());
  connect(inputBox,SIGNAL(currentIndexChanged(int)),pipe,SLOT(update(int)));
}

QtDialog::MoleculesQtQuery::~MoleculesQtQuery()
{
  delete pipe;
}

// Handling is easy, since the GUI makes it impossible to select invalid values
bool QtDialog::MoleculesQtQuery::handle()
{
  return true;
}

QtDialog::VectorQtQuery::VectorQtQuery(std::string title, bool _check,QBoxLayout *_parent,QtDialog *_dialog) :
    Dialog::VectorQuery(title,_check),
    parent(_parent)
{
  mainLayout= new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));
  mainLayout->addWidget(titleLabel);
  subLayout = new QVBoxLayout();
  mainLayout->addLayout(subLayout);
  QComboBox* inputBox = new QComboBox();
  coordLayout = new QHBoxLayout();
  subLayout->addLayout(coordLayout);
  coordLabel = new QLabel(QString("x,y,z"));
  coordLayout->addWidget(coordLabel);
  coordInput = new QDoubleSpinBox();
//  coordInput->setRange(0,M.at(i,i));
  coordInput->setDecimals(3);
  coordLayout->addWidget(coordInput);
  pipe = new VectorQtQueryPipe(&(tmp),_dialog,inputBox);
  //pipe->update(coordInput->value());
  connect(coordInput,SIGNAL(valueChanged(double)),pipe,SLOT(update(double)));
  parent->addLayout(mainLayout);
}

QtDialog::VectorQtQuery::~VectorQtQuery()
{}

bool QtDialog::VectorQtQuery::handle() {
  return true;
}


QtDialog::VectorsQtQuery::VectorsQtQuery(std::string title, bool _check,QBoxLayout *_parent,QtDialog *_dialog) :
    Dialog::VectorsQuery(title,_check),
    parent(_parent)
{
  mainLayout= new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));
  mainLayout->addWidget(titleLabel);
  subLayout = new QVBoxLayout();
  mainLayout->addLayout(subLayout);
  QComboBox* inputBox = new QComboBox();
  coordLayout = new QHBoxLayout();
  subLayout->addLayout(coordLayout);
  coordLabel = new QLabel(QString("x,y,z"));
  coordLayout->addWidget(coordLabel);
  coordInput = new QDoubleSpinBox();
//  coordInput->setRange(0,M.at(i,i));
  coordInput->setDecimals(3);
  coordLayout->addWidget(coordInput);
  pipe = new VectorsQtQueryPipe(&(tmp),_dialog,inputBox);
  //pipe->update(coordInput->value());
  connect(coordInput,SIGNAL(valueChanged(double)),pipe,SLOT(update(double)));
  parent->addLayout(mainLayout);
}

QtDialog::VectorsQtQuery::~VectorsQtQuery()
{}

bool QtDialog::VectorsQtQuery::handle() {
  return true;
}


QtDialog::ElementQtQuery::ElementQtQuery(std::string _title, QBoxLayout *_parent, QtDialog *_dialog) :
    Dialog::ElementQuery(_title),
    parent(_parent)
{
  periodentafel *periode = World::getInstance().getPeriode();
  thisLayout = new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));
  inputBox = new QComboBox();
  for(periodentafel::const_iterator iter = periode->begin();
      iter!=periode->end();
      ++iter)
  {
    std::stringstream sstr;
    sstr << (*iter).first << "\t" << (*iter).second->getName();
    inputBox->addItem(QString(sstr.str().c_str()),QVariant((*iter).first));
  }
  parent->addLayout(thisLayout);
  thisLayout->addWidget(titleLabel);
  thisLayout->addWidget(inputBox);

  pipe = new ElementQtQueryPipe(&tmp,_dialog,inputBox);
  pipe->update(inputBox->currentIndex());
  connect(inputBox,SIGNAL(currentIndexChanged(int)),pipe,SLOT(update(int)));
}

QtDialog::ElementQtQuery::~ElementQtQuery()
{
  delete pipe;
}

bool QtDialog::ElementQtQuery::handle(){
  return true;
}


QtDialog::ElementsQtQuery::ElementsQtQuery(std::string _title, QBoxLayout *_parent, QtDialog *_dialog) :
    Dialog::ElementsQuery(_title),
    parent(_parent)
{
  periodentafel *periode = World::getInstance().getPeriode();
  thisLayout = new QHBoxLayout();
  titleLabel = new QLabel(QString(getTitle().c_str()));
  inputBox = new QComboBox();
  for(periodentafel::const_iterator iter = periode->begin();
      iter!=periode->end();
      ++iter)
  {
    std::stringstream sstr;
    sstr << (*iter).first << "\t" << (*iter).second->getName();
    inputBox->addItem(QString(sstr.str().c_str()),QVariant((*iter).first));
  }
  parent->addLayout(thisLayout);
  thisLayout->addWidget(titleLabel);
  thisLayout->addWidget(inputBox);

  pipe = new ElementsQtQueryPipe(&tmp,_dialog,inputBox);
  pipe->update(inputBox->currentIndex());
  connect(inputBox,SIGNAL(currentIndexChanged(int)),pipe,SLOT(update(int)));
}

QtDialog::ElementsQtQuery::~ElementsQtQuery()
{
  delete pipe;
}

bool QtDialog::ElementsQtQuery::handle(){
  return true;
}

QtDialog::EmptyQtQuery::EmptyQtQuery(std::string _title, QBoxLayout *_parent, QtDialog *_dialog) :
    Dialog::EmptyQuery(_title),
    parent(_parent)
{
  titleLabel = new QLabel(QString(getTitle().c_str()));

  parent->addLayout(thisLayout);
  thisLayout->addWidget(titleLabel);

  pipe = new EmptyQtQueryPipe(_dialog,titleLabel);
}

QtDialog::EmptyQtQuery::~EmptyQtQuery()
{
  delete pipe;
}

bool QtDialog::EmptyQtQuery::handle(){
  return true;
}


QtDialog::BooleanQtQuery::BooleanQtQuery(std::string _title, QBoxLayout *_parent, QtDialog *_dialog) :
    Dialog::BooleanQuery(_title),
    parent(_parent)
{
  titleLabel = new QLabel(QString(getTitle().c_str()),_dialog);
  booleanComboBox = new QComboBox(_dialog);
  booleanComboBox->insertItem(1, QString("true"));
  booleanComboBox->insertItem(0, QString("false"));

  parent->addLayout(thisLayout);
  thisLayout->addWidget(titleLabel);
  thisLayout->addWidget(booleanComboBox);

  pipe = new BooleanQtQueryPipe(&tmp,_dialog,booleanComboBox);
  connect(booleanComboBox, SIGNAL(currentIndexChanged()), pipe, SLOT(update()));
}

QtDialog::BooleanQtQuery::~BooleanQtQuery()
{
  delete pipe;
}

bool QtDialog::BooleanQtQuery::handle(){
  return true;
}


QtDialog::FileQtQuery::FileQtQuery(std::string _title, QBoxLayout *_parent, QtDialog *_dialog) :
    Dialog::FileQuery(_title),
    parent(_parent)
{

  filenameLineEdit = new QLineEdit(_dialog);
  filenameLineEdit->setText(QString());
  filenameLineEdit->setReadOnly(true);

  filenameLabel = new QLabel(QString("Input file:"));
  filenameLabel->setBuddy(filenameLineEdit);

  filedialogButton = new QPushButton("&Choose", _dialog);

  pipe = new FileQtQueryPipe(&tmp,_dialog,filenameLineEdit,filedialogButton);

  thisLayout = new QHBoxLayout();
  parent->addLayout(thisLayout);
  thisLayout->addWidget(filenameLabel);
  thisLayout->addWidget(filenameLineEdit);
  thisLayout->addWidget(filedialogButton);

  QObject::connect(filedialogButton,SIGNAL(clicked()),pipe,SLOT(showFileDialog()));
}

QtDialog::FileQtQuery::~FileQtQuery()
{
  delete pipe;
}

bool QtDialog::FileQtQuery::handle(){
  return true;
}

/*************************** Plumbing *******************************/


template<typename T> QtQueryListPipe<T>::QtQueryListPipe(std::vector<T> *_content, QtDialog *_dialog, QLineEdit *_inputBox, QListWidget *_inputList, QPushButton *_AddButton, QPushButton *_RemoveButton) :
  content(_content),
  dialog(_dialog),
  inputBox(_inputBox),
  inputList(_inputList),
  AddButton(_AddButton),
  RemoveButton(_RemoveButton)
{}

template<typename T> QtQueryListPipe<T>::~QtQueryListPipe()
{}

template<typename T> void QtQueryListPipe<T>::IntegerEntered(const QString&)
{
  AddButton->setEnabled(true);
}

template<typename T> void QtQueryListPipe<T>::IntegerSelected()
{
  if (inputList->selectedItems().empty())
    RemoveButton->setEnabled(false);
  else
    RemoveButton->setEnabled(true);
}

template<typename T> void QtQueryListPipe<T>::AddInteger() {
  // type-check
  std::string text = inputBox->text().toStdString();
  int number = 0;
  try {
    number = boost::lexical_cast<int>(text);
  } catch (boost::bad_lexical_cast&) {
    return;
  };
  // add item to both
  inputList->addItem(QString(number));
  AddValue(number);
}

template<typename T> void QtQueryListPipe<T>::AddValue(T item) {
  content->push_back(item);

  dialog->update();
}

template<typename T> void QtQueryListPipe<T>::RemoveInteger() {
  QList<QListWidgetItem *> items = inputList->selectedItems();
  for (QList<QListWidgetItem *>::iterator iter = items.begin(); !items.empty(); iter = items.begin()) {
    // obtain which position item has (by making it current item)
    inputList->setCurrentItem(*iter);
    // remove
    QtQueryListPipe<T>::RemoteRow(inputList->currentRow()); // template parameters needs to be known, such that compiler knows which to call
    inputList->removeItemWidget(*iter);
  }
}

template<typename T> void QtQueryListPipe<T>::RemoveRow(int row) {
  int counter = 0;
  typename std::vector<T>::iterator iter = content->begin();
  for (; iter != content->end(); ++iter)
    if (counter++ == row)
      break;
  if (iter != content->end())
      content->erase(iter);
}


StringQtQueryPipe::StringQtQueryPipe(string *_content, QtDialog *_dialog) :
  content(_content),
  dialog(_dialog)
{}

StringQtQueryPipe::~StringQtQueryPipe()
{}

void StringQtQueryPipe::update(const QString& newText) {
  content->assign(newText.toStdString());
  dialog->update();
}

IntQtQueryPipe::IntQtQueryPipe(int *_content, QtDialog *_dialog) :
  content(_content),
  dialog(_dialog)
{}

IntQtQueryPipe::~IntQtQueryPipe()
{}

void IntQtQueryPipe::update(int newInt) {
  (*content) = newInt;
  dialog->update();
}

DoubleQtQueryPipe::DoubleQtQueryPipe(double *_content, QtDialog *_dialog) :
  content(_content),
  dialog(_dialog)
{}

DoubleQtQueryPipe::~DoubleQtQueryPipe()
{}

void DoubleQtQueryPipe::update(double newDbl) {
  (*content) = newDbl;
  dialog->update();
}

VectorQtQueryPipe::VectorQtQueryPipe(Vector *_content, QtDialog *_dialog, QComboBox *_theBox) :
  content(_content),
  dialog(_dialog),
  theBox(_theBox)
{}

VectorQtQueryPipe::~VectorQtQueryPipe()
{}

void VectorQtQueryPipe::update() {
  dialog->update();
}

VectorsQtQueryPipe::VectorsQtQueryPipe(std::vector<Vector> *_content, QtDialog *_dialog, QComboBox *_theBox) :
  content(_content),
  dialog(_dialog),
  theBox(_theBox)
{}

VectorsQtQueryPipe::~VectorsQtQueryPipe()
{}

void VectorsQtQueryPipe::update() {
  dialog->update();
}

BoxQtQueryPipe::BoxQtQueryPipe(Box &_content, QtDialog *_dialog, QTableWidget *_inputTable) :
  content(_content),
  dialog(_dialog),
  inputTable(_inputTable)
{
  tmpM = new Matrix();
  tmpM->zero();
}

BoxQtQueryPipe::~BoxQtQueryPipe()
{
  delete tmpM;
}

void BoxQtQueryPipe::update(int row, int column)
{
  tmpM->at(row, column) = inputTable->item(row, column)->text().toDouble();
  content.setM(*tmpM);
  dialog->update();
}


AtomQtQueryPipe::AtomQtQueryPipe(atom **_content, QtDialog *_dialog, QComboBox *_theBox) :
  content(_content),
  dialog(_dialog),
  theBox(_theBox)
{}

AtomQtQueryPipe::~AtomQtQueryPipe()
{}

void AtomQtQueryPipe::update(int newIndex) {
  QVariant data = theBox->itemData(newIndex);
  int idx = data.toInt();
  (*content) = World::getInstance().getAtom(AtomById(idx));
  dialog->update();
}


AtomsQtQueryPipe::AtomsQtQueryPipe(std::vector<atom *>*_content, QtDialog *_dialog, QListWidget *_theList) :
  content(_content),
  dialog(_dialog),
  theList(_theList)
{}

AtomsQtQueryPipe::~AtomsQtQueryPipe()
{}

void AtomsQtQueryPipe::update() {
  // clear target and put all atoms therein
  (*content).clear();
  for (std::set<atom *>::iterator iter = currentList.begin(); iter != currentList.end(); ++iter)
    (*content).push_back(*iter);
  dialog->update();
}

void AtomsQtQueryPipe::add() {
  QList<QListWidgetItem *> items = theList->selectedItems();
  for (QList<QListWidgetItem *>::iterator iter = items.begin();iter != items.end();++iter) {
    const int index = (*iter)->text().toInt();
    atom *Walker = World::getInstance().getAtom(AtomById(index));
    if (Walker) {
      (*content).push_back(Walker);
      currentList.insert(Walker);
      if (lookup.find(index) != lookup.end())
        lookup.insert(pair<int, atom*>(index, Walker));
    }
  }
  update();
}

void AtomsQtQueryPipe::remove() {
  QList<QListWidgetItem *> items = theList->selectedItems();
  for (QList<QListWidgetItem *>::iterator iter = items.begin();iter != items.end();++iter) {
    const int index = (*iter)->text().toInt();
    atom *Walker = World::getInstance().getAtom(AtomById(index));
    if (Walker) {
      currentList.erase(Walker);
    }
  }
  update();
}


MoleculeQtQueryPipe::MoleculeQtQueryPipe(molecule **_content, QtDialog *_dialog, QComboBox *_theBox) :
  content(_content),
  dialog(_dialog),
  theBox(_theBox)
{}

MoleculeQtQueryPipe::~MoleculeQtQueryPipe()
{}

void MoleculeQtQueryPipe::update(int newIndex) {
  QVariant data = theBox->itemData(newIndex);
  int idx = data.toInt();
  (*content) = World::getInstance().getMolecule(MoleculeById(idx));
  dialog->update();
}


MoleculesQtQueryPipe::MoleculesQtQueryPipe(std::vector<molecule *>*_content, QtDialog *_dialog, QComboBox *_theBox) :
  content(_content),
  dialog(_dialog),
  theBox(_theBox)
{}

MoleculesQtQueryPipe::~MoleculesQtQueryPipe()
{}

void MoleculesQtQueryPipe::update(int newIndex) {
  QVariant data = theBox->itemData(newIndex);
  int idx = data.toInt();
  molecule *mol = World::getInstance().getMolecule(MoleculeById(idx));
  if (mol)
    (*content).push_back(mol);
  dialog->update();
}

ElementQtQueryPipe::ElementQtQueryPipe(const element **_content, QtDialog *_dialog, QComboBox *_theBox) :
  content(_content),
  dialog(_dialog),
  theBox(_theBox)
{}

ElementQtQueryPipe::~ElementQtQueryPipe()
{}

void ElementQtQueryPipe::update(int newIndex) {
  QVariant data = theBox->itemData(newIndex);
  int idx = data.toInt();
  *content = World::getInstance().getPeriode()->FindElement(idx);
  dialog->update();
}

ElementsQtQueryPipe::ElementsQtQueryPipe(std::vector<const element *>*_content, QtDialog *_dialog, QComboBox *_theBox) :
  content(_content),
  dialog(_dialog),
  theBox(_theBox)
{}

ElementsQtQueryPipe::~ElementsQtQueryPipe()
{}

void ElementsQtQueryPipe::update(int newIndex) {
  QVariant data = theBox->itemData(newIndex);
  int idx = data.toInt();
  const element *elemental = World::getInstance().getPeriode()->FindElement(idx);
  if(elemental)
    (*content).push_back(elemental);
  dialog->update();
}

EmptyQtQueryPipe::EmptyQtQueryPipe(QtDialog *_dialog, QLabel *_textLabel) :
  dialog(_dialog),
  textLabel(_textLabel)
{}

EmptyQtQueryPipe::~EmptyQtQueryPipe()
{}

void EmptyQtQueryPipe::update() {
  dialog->update();
}

BooleanQtQueryPipe::BooleanQtQueryPipe(const bool *_content, QtDialog *_dialog, QComboBox *_booleanComboBox) :
  content(_content),
  dialog(_dialog),
  booleanComboBox(_booleanComboBox)
{}

BooleanQtQueryPipe::~BooleanQtQueryPipe()
{}

void BooleanQtQueryPipe::update() {
  dialog->update();
}

FileQtQueryPipe::FileQtQueryPipe(boost::filesystem::path *_content, QtDialog *_dialog, QLineEdit *_filenameLineEdit, QPushButton *_filedialogButton) :
  content(_content),
  dialog(_dialog),
  filenameLineEdit(_filenameLineEdit),
  filedialogButton(_filedialogButton)
{
  theFileDialog = NULL;
}

FileQtQueryPipe::~FileQtQueryPipe()
{
  if (theFileDialog != NULL)
    delete theFileDialog;
}

void FileQtQueryPipe::update() {
  QStringList ListOfFilenames = theFileDialog->selectedFiles();
  std::cout << "Selected File is " << ListOfFilenames.at(0).toStdString() << std::endl;
  (*content) = ListOfFilenames.at(0).toStdString();
  filenameLineEdit->setText(QString::fromStdString((*content).string()));
  dialog->update();
}

void FileQtQueryPipe::showFileDialog() {
  filedialogButton->setFlat(true);
  if (theFileDialog == NULL) {
    theFileDialog = new QFileDialog(NULL, tr("Open input file"), QString(), tr("ParallelCarParrinello (*.conf);;MassivelyParallelQuantumChemistry (*.mpqc);;ParticleDataBase (*.pdb);;XYZ (*.xyz)"));
    theFileDialog->setAcceptMode(QFileDialog::AcceptOpen);
    theFileDialog->setFileMode(QFileDialog::ExistingFile);
    theFileDialog->setViewMode(QFileDialog::List);
  }
  theFileDialog->exec();

  update();
  filedialogButton->setFlat(false);
}



